<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_localhost" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "Agent_Master", "varname" => "firefoxbd_at_localhost__Agent_Master", "altvarname" => "Agent_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Assets_Group", "varname" => "firefoxbd_at_localhost__Assets_Group", "altvarname" => "Assets_Group", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Assets_Item", "varname" => "firefoxbd_at_localhost__Assets_Item", "altvarname" => "Assets_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Model", "varname" => "firefoxbd_at_localhost__BD_Model", "altvarname" => "BD_Model", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_SHPG_IBWS_Remarks", "varname" => "firefoxbd_at_localhost__BD_SHPG_IBWS_Remarks", "altvarname" => "BD_SHPG_IBWS_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Container_Size", "varname" => "firefoxbd_at_localhost__Container_Size", "altvarname" => "Container_Size", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "corvoproject_audit", "varname" => "firefoxbd_at_localhost__corvoproject_audit", "altvarname" => "corvoproject_audit", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Courier_Info_Head", "varname" => "firefoxbd_at_localhost__Courier_Info_Head", "altvarname" => "Courier_Info_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Courier_Info_Item", "varname" => "firefoxbd_at_localhost__Courier_Info_Item", "altvarname" => "Courier_Info_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CTN_Bonds", "varname" => "firefoxbd_at_localhost__CTN_Bonds", "altvarname" => "CTN_Bonds", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CusDec", "varname" => "firefoxbd_at_localhost__CusDec", "altvarname" => "CusDec", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CusDec_Pclist", "varname" => "firefoxbd_at_localhost__CusDec_Pclist", "altvarname" => "CusDec_Pclist", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer", "varname" => "firefoxbd_at_localhost__Customer", "altvarname" => "Customer", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Destinations", "varname" => "firefoxbd_at_localhost__Destinations", "altvarname" => "Destinations", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Cont", "varname" => "firefoxbd_at_localhost__Exp_Inv_Cont", "altvarname" => "Exp_Inv_Cont", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Ex_Item", "varname" => "firefoxbd_at_localhost__Exp_Inv_Ex_Item", "altvarname" => "Exp_Inv_Ex_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Ex_Pack", "varname" => "firefoxbd_at_localhost__Exp_Inv_Ex_Pack", "altvarname" => "Exp_Inv_Ex_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Head", "varname" => "firefoxbd_at_localhost__Exp_Inv_Head", "altvarname" => "Exp_Inv_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Item", "varname" => "firefoxbd_at_localhost__Exp_Inv_Item", "altvarname" => "Exp_Inv_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Main", "varname" => "firefoxbd_at_localhost__Exp_Inv_Main", "altvarname" => "Exp_Inv_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Main_Remarks", "varname" => "firefoxbd_at_localhost__Exp_Inv_Main_Remarks", "altvarname" => "Exp_Inv_Main_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Pack", "varname" => "firefoxbd_at_localhost__Exp_Inv_Pack", "altvarname" => "Exp_Inv_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Remarks", "varname" => "firefoxbd_at_localhost__Exp_Inv_Remarks", "altvarname" => "Exp_Inv_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Export_Advice_Item", "varname" => "firefoxbd_at_localhost__Export_Advice_Item", "altvarname" => "Export_Advice_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Export_Coststatement_Head", "varname" => "firefoxbd_at_localhost__Export_Coststatement_Head", "altvarname" => "Export_Coststatement_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Export_SCCont", "varname" => "firefoxbd_at_localhost__Export_SCCont", "altvarname" => "Export_SCCont", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Export_Shipping_Advise", "varname" => "firefoxbd_at_localhost__Export_Shipping_Advise", "altvarname" => "Export_Shipping_Advise", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Factory", "varname" => "firefoxbd_at_localhost__Factory", "altvarname" => "Factory", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_uggroups", "varname" => "firefoxbd_at_localhost__ffxbderp_uggroups", "altvarname" => "ffxbderp_uggroups", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugmembers", "varname" => "firefoxbd_at_localhost__ffxbderp_ugmembers", "altvarname" => "ffxbderp_ugmembers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugrights", "varname" => "firefoxbd_at_localhost__ffxbderp_ugrights", "altvarname" => "ffxbderp_ugrights", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "FGS_Reason", "varname" => "firefoxbd_at_localhost__FGS_Reason", "altvarname" => "FGS_Reason", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "FGS_Stock", "varname" => "firefoxbd_at_localhost__FGS_Stock", "altvarname" => "FGS_Stock", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Forwaders", "varname" => "firefoxbd_at_localhost__Forwaders", "altvarname" => "Forwaders", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Forwaders_Local", "varname" => "firefoxbd_at_localhost__Forwaders_Local", "altvarname" => "Forwaders_Local", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Gatepass_Head", "varname" => "firefoxbd_at_localhost__Gatepass_Head", "altvarname" => "Gatepass_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Gatepass_Item", "varname" => "firefoxbd_at_localhost__Gatepass_Item", "altvarname" => "Gatepass_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_Inv", "varname" => "firefoxbd_at_localhost__Ibws_Head_Inv", "altvarname" => "Ibws_Head_Inv", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Inv_Plus_Minus2", "varname" => "firefoxbd_at_localhost__Ibws_Inv_Plus_Minus2", "altvarname" => "Ibws_Inv_Plus_Minus2", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Import_Items", "varname" => "firefoxbd_at_localhost__Import_Items", "altvarname" => "Import_Items", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Loading_Head", "varname" => "firefoxbd_at_localhost__Loading_Head", "altvarname" => "Loading_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Loading_Ports", "varname" => "firefoxbd_at_localhost__Loading_Ports", "altvarname" => "Loading_Ports", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Parts", "varname" => "firefoxbd_at_localhost__Parts", "altvarname" => "Parts", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Payterms_new", "varname" => "firefoxbd_at_localhost__Payterms_new", "altvarname" => "Payterms_new", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PC_Head", "varname" => "firefoxbd_at_localhost__PC_Head", "altvarname" => "PC_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PC_Item", "varname" => "firefoxbd_at_localhost__PC_Item", "altvarname" => "PC_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pc_Terms", "varname" => "firefoxbd_at_localhost__Pc_Terms", "altvarname" => "Pc_Terms", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_B_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_B_Head", "altvarname" => "PO_Imp_B_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_F_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_F_Head", "altvarname" => "PO_Imp_F_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Request_Exp_Permit_Head", "varname" => "firefoxbd_at_localhost__Request_Exp_Permit_Head", "altvarname" => "Request_Exp_Permit_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Request_Exp_Permit_Item", "varname" => "firefoxbd_at_localhost__Request_Exp_Permit_Item", "altvarname" => "Request_Exp_Permit_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Request_Imp_Permit_cont", "varname" => "firefoxbd_at_localhost__Request_Imp_Permit_cont", "altvarname" => "Request_Imp_Permit_cont", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Request_Imp_Permit_Head", "varname" => "firefoxbd_at_localhost__Request_Imp_Permit_Head", "altvarname" => "Request_Imp_Permit_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Request_Imp_Permit_Item", "varname" => "firefoxbd_at_localhost__Request_Imp_Permit_Item", "altvarname" => "Request_Imp_Permit_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Head", "varname" => "firefoxbd_at_localhost__Sc_Head", "altvarname" => "Sc_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SC_Item", "varname" => "firefoxbd_at_localhost__SC_Item", "altvarname" => "SC_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sh_Company", "varname" => "firefoxbd_at_localhost__Sh_Company", "altvarname" => "Sh_Company", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "shipping_head_contlist", "varname" => "firefoxbd_at_localhost__shipping_head_contlist", "altvarname" => "shipping_head_contlist", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SHP_Adv_Data", "varname" => "firefoxbd_at_localhost__SHP_Adv_Data", "altvarname" => "SHP_Adv_Data", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SHP_Adv_Head", "varname" => "firefoxbd_at_localhost__SHP_Adv_Head", "altvarname" => "SHP_Adv_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SHP_Adv_Remarks", "varname" => "firefoxbd_at_localhost__SHP_Adv_Remarks", "altvarname" => "SHP_Adv_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Item", "varname" => "firefoxbd_at_localhost__ST_Item", "altvarname" => "ST_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Title", "varname" => "firefoxbd_at_localhost__Title", "altvarname" => "Title", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "TRI_Exp_Inv_Main", "varname" => "firefoxbd_at_localhost__TRI_Exp_Inv_Main", "altvarname" => "TRI_Exp_Inv_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "webusers", "varname" => "firefoxbd_at_localhost__webusers", "altvarname" => "webusers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>