<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_localhost" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "Att_Dailly", "varname" => "firefoxbd_at_localhost__Att_Dailly", "altvarname" => "Att_Dailly", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Att_Dailly_Item", "varname" => "firefoxbd_at_localhost__Att_Dailly_Item", "altvarname" => "Att_Dailly_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Att_Department_Emp", "varname" => "firefoxbd_at_localhost__Att_Department_Emp", "altvarname" => "Att_Department_Emp", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Att_Departments", "varname" => "firefoxbd_at_localhost__Att_Departments", "altvarname" => "Att_Departments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Att_Members", "varname" => "firefoxbd_at_localhost__Att_Members", "altvarname" => "Att_Members", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Att_Remarks", "varname" => "firefoxbd_at_localhost__Att_Remarks", "altvarname" => "Att_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Holidays", "varname" => "firefoxbd_at_localhost__BD_Holidays", "altvarname" => "BD_Holidays", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Meetings", "varname" => "firefoxbd_at_localhost__BD_Meetings", "altvarname" => "BD_Meetings", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Model", "varname" => "firefoxbd_at_localhost__BD_Model", "altvarname" => "BD_Model", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Model_Refer", "varname" => "firefoxbd_at_localhost__BD_Model_Refer", "altvarname" => "BD_Model_Refer", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Comments_Shortage", "varname" => "firefoxbd_at_localhost__Comments_Shortage", "altvarname" => "Comments_Shortage", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Comments_Shortage_Bulk", "varname" => "firefoxbd_at_localhost__Comments_Shortage_Bulk", "altvarname" => "Comments_Shortage_Bulk", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "corvoproject_audit", "varname" => "firefoxbd_at_localhost__corvoproject_audit", "altvarname" => "corvoproject_audit", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CRTN_Mesurement", "varname" => "firefoxbd_at_localhost__CRTN_Mesurement", "altvarname" => "CRTN_Mesurement", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CusDec", "varname" => "firefoxbd_at_localhost__CusDec", "altvarname" => "CusDec", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer", "varname" => "firefoxbd_at_localhost__Customer", "altvarname" => "Customer", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Defect_Action", "varname" => "firefoxbd_at_localhost__Defect_Action", "altvarname" => "Defect_Action", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Defect_Analysis_Head", "varname" => "firefoxbd_at_localhost__Defect_Analysis_Head", "altvarname" => "Defect_Analysis_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Defect_Analysis_Item", "varname" => "firefoxbd_at_localhost__Defect_Analysis_Item", "altvarname" => "Defect_Analysis_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Defect_Departments", "varname" => "firefoxbd_at_localhost__Defect_Departments", "altvarname" => "Defect_Departments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Defect_Emp_Master", "varname" => "firefoxbd_at_localhost__Defect_Emp_Master", "altvarname" => "Defect_Emp_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Defect_Master", "varname" => "firefoxbd_at_localhost__Defect_Master", "altvarname" => "Defect_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_uggroups", "varname" => "firefoxbd_at_localhost__ffxbderp_uggroups", "altvarname" => "ffxbderp_uggroups", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugmembers", "varname" => "firefoxbd_at_localhost__ffxbderp_ugmembers", "altvarname" => "ffxbderp_ugmembers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugrights", "varname" => "firefoxbd_at_localhost__ffxbderp_ugrights", "altvarname" => "ffxbderp_ugrights", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "FGS_Adj", "varname" => "firefoxbd_at_localhost__FGS_Adj", "altvarname" => "FGS_Adj", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Fright", "varname" => "firefoxbd_at_localhost__Fright", "altvarname" => "Fright", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Local_Sales", "varname" => "firefoxbd_at_localhost__Local_Sales", "altvarname" => "Local_Sales", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Local_Sales_Item", "varname" => "firefoxbd_at_localhost__Local_Sales_Item", "altvarname" => "Local_Sales_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "OT_Emp_Time", "varname" => "firefoxbd_at_localhost__OT_Emp_Time", "altvarname" => "OT_Emp_Time", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "OT_Master", "varname" => "firefoxbd_at_localhost__OT_Master", "altvarname" => "OT_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pro_Checklist_ConsiderItems", "varname" => "firefoxbd_at_localhost__Pro_Checklist_ConsiderItems", "altvarname" => "Pro_Checklist_ConsiderItems", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pro_Checklist_Departments", "varname" => "firefoxbd_at_localhost__Pro_Checklist_Departments", "altvarname" => "Pro_Checklist_Departments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pro_Checklist_Head", "varname" => "firefoxbd_at_localhost__Pro_Checklist_Head", "altvarname" => "Pro_Checklist_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pro_Checklist_IBWS", "varname" => "firefoxbd_at_localhost__Pro_Checklist_IBWS", "altvarname" => "Pro_Checklist_IBWS", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pro_Checklist_ProdDates", "varname" => "firefoxbd_at_localhost__Pro_Checklist_ProdDates", "altvarname" => "Pro_Checklist_ProdDates", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "QC_Comments_Head", "varname" => "firefoxbd_at_localhost__QC_Comments_Head", "altvarname" => "QC_Comments_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "QC_Comments_Item", "varname" => "firefoxbd_at_localhost__QC_Comments_Item", "altvarname" => "QC_Comments_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ref_Procheck", "varname" => "firefoxbd_at_localhost__Ref_Procheck", "altvarname" => "Ref_Procheck", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Reminder_Head_BD", "varname" => "firefoxbd_at_localhost__Reminder_Head_BD", "altvarname" => "Reminder_Head_BD", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Replacement_Item", "varname" => "firefoxbd_at_localhost__Replacement_Item", "altvarname" => "Replacement_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Replacement_Item_BD", "varname" => "firefoxbd_at_localhost__Replacement_Item_BD", "altvarname" => "Replacement_Item_BD", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Replcement_Req_Head", "varname" => "firefoxbd_at_localhost__Replcement_Req_Head", "altvarname" => "Replcement_Req_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Head", "varname" => "firefoxbd_at_localhost__Sc_Head", "altvarname" => "Sc_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SC_Item", "varname" => "firefoxbd_at_localhost__SC_Item", "altvarname" => "SC_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Location", "varname" => "firefoxbd_at_localhost__ST_Item_Location", "altvarname" => "ST_Item_Location", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "To_Time", "varname" => "firefoxbd_at_localhost__To_Time", "altvarname" => "To_Time", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "webusers", "varname" => "firefoxbd_at_localhost__webusers", "altvarname" => "webusers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Wip_Departments", "varname" => "firefoxbd_at_localhost__Wip_Departments", "altvarname" => "Wip_Departments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Wip_Head", "varname" => "firefoxbd_at_localhost__Wip_Head", "altvarname" => "Wip_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Wip_Paint_Departments", "varname" => "firefoxbd_at_localhost__Wip_Paint_Departments", "altvarname" => "Wip_Paint_Departments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Wip_Paint_Head", "varname" => "firefoxbd_at_localhost__Wip_Paint_Head", "altvarname" => "Wip_Paint_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Wip_Paint_TimeTable", "varname" => "firefoxbd_at_localhost__Wip_Paint_TimeTable", "altvarname" => "Wip_Paint_TimeTable", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Wip_TimeTable", "varname" => "firefoxbd_at_localhost__Wip_TimeTable", "altvarname" => "Wip_TimeTable", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>