<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_localhost" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "AFKT_Item", "varname" => "firefoxbd_at_localhost__AFKT_Item", "altvarname" => "AFKT_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Agent_Master", "varname" => "firefoxbd_at_localhost__Agent_Master", "altvarname" => "Agent_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Agent_Master_Fty", "varname" => "firefoxbd_at_localhost__Agent_Master_Fty", "altvarname" => "Agent_Master_Fty", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Applications", "varname" => "firefoxbd_at_localhost__Applications", "altvarname" => "Applications", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Artwork_Master", "varname" => "firefoxbd_at_localhost__Artwork_Master", "altvarname" => "Artwork_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Bike_Model", "varname" => "firefoxbd_at_localhost__BD_Bike_Model", "altvarname" => "BD_Bike_Model", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Meetings", "varname" => "firefoxbd_at_localhost__BD_Meetings", "altvarname" => "BD_Meetings", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Model", "varname" => "firefoxbd_at_localhost__BD_Model", "altvarname" => "BD_Model", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Shipping_Advice", "varname" => "firefoxbd_at_localhost__BD_Shipping_Advice", "altvarname" => "BD_Shipping_Advice", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BL_Cont_Inv", "varname" => "firefoxbd_at_localhost__BL_Cont_Inv", "altvarname" => "BL_Cont_Inv", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BL_Cont_Item", "varname" => "firefoxbd_at_localhost__BL_Cont_Item", "altvarname" => "BL_Cont_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BL_Head", "varname" => "firefoxbd_at_localhost__BL_Head", "altvarname" => "BL_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BL_Shipper", "varname" => "firefoxbd_at_localhost__BL_Shipper", "altvarname" => "BL_Shipper", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_DaillyItems", "varname" => "firefoxbd_at_localhost__Cash_Flow_DaillyItems", "altvarname" => "Cash_Flow_DaillyItems", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_MainItems", "varname" => "firefoxbd_at_localhost__Cash_Flow_MainItems", "altvarname" => "Cash_Flow_MainItems", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Category", "varname" => "firefoxbd_at_localhost__Category", "altvarname" => "Category", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cert_Content", "varname" => "firefoxbd_at_localhost__Cert_Content", "altvarname" => "Cert_Content", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cert_Name", "varname" => "firefoxbd_at_localhost__Cert_Name", "altvarname" => "Cert_Name", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CERT_Nutral", "varname" => "firefoxbd_at_localhost__CERT_Nutral", "altvarname" => "CERT_Nutral", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cert_Title", "varname" => "firefoxbd_at_localhost__Cert_Title", "altvarname" => "Cert_Title", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Comp_Item", "varname" => "firefoxbd_at_localhost__Comp_Item", "altvarname" => "Comp_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Container_Data", "varname" => "firefoxbd_at_localhost__Container_Data", "altvarname" => "Container_Data", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Container_Head", "varname" => "firefoxbd_at_localhost__Container_Head", "altvarname" => "Container_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "corvoproject_audit", "varname" => "firefoxbd_at_localhost__corvoproject_audit", "altvarname" => "corvoproject_audit", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_China_Sea_Frt", "varname" => "firefoxbd_at_localhost__Cost_China_Sea_Frt", "altvarname" => "Cost_China_Sea_Frt", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Ckd", "varname" => "firefoxbd_at_localhost__Cost_Ckd", "altvarname" => "Cost_Ckd", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Customer", "varname" => "firefoxbd_at_localhost__Cost_Customer", "altvarname" => "Cost_Customer", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Dep", "varname" => "firefoxbd_at_localhost__Cost_Dep", "altvarname" => "Cost_Dep", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Fixed", "varname" => "firefoxbd_at_localhost__Cost_Fixed", "altvarname" => "Cost_Fixed", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Main", "varname" => "firefoxbd_at_localhost__Cost_Main", "altvarname" => "Cost_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Paint_New", "varname" => "firefoxbd_at_localhost__Cost_Paint_New", "altvarname" => "Cost_Paint_New", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Pkg", "varname" => "firefoxbd_at_localhost__Cost_Pkg", "altvarname" => "Cost_Pkg", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Rimmaterials", "varname" => "firefoxbd_at_localhost__Cost_Rimmaterials", "altvarname" => "Cost_Rimmaterials", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_TTT_New", "varname" => "firefoxbd_at_localhost__Cost_TTT_New", "altvarname" => "Cost_TTT_New", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_UP_Down", "varname" => "firefoxbd_at_localhost__Cost_UP_Down", "altvarname" => "Cost_UP_Down", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cost_Var_OH", "varname" => "firefoxbd_at_localhost__Cost_Var_OH", "altvarname" => "Cost_Var_OH", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Country", "varname" => "firefoxbd_at_localhost__Country", "altvarname" => "Country", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Currency", "varname" => "firefoxbd_at_localhost__Currency", "altvarname" => "Currency", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CusDec", "varname" => "firefoxbd_at_localhost__CusDec", "altvarname" => "CusDec", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer", "varname" => "firefoxbd_at_localhost__Customer", "altvarname" => "Customer", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer_Comments", "varname" => "firefoxbd_at_localhost__Customer_Comments", "altvarname" => "Customer_Comments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer_Create_SC", "varname" => "firefoxbd_at_localhost__Customer_Create_SC", "altvarname" => "Customer_Create_SC", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer_Pkg_Report", "varname" => "firefoxbd_at_localhost__Customer_Pkg_Report", "altvarname" => "Customer_Pkg_Report", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Decimal_Val", "varname" => "firefoxbd_at_localhost__Decimal_Val", "altvarname" => "Decimal_Val", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Departments", "varname" => "firefoxbd_at_localhost__Departments", "altvarname" => "Departments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Destinations", "varname" => "firefoxbd_at_localhost__Destinations", "altvarname" => "Destinations", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ex_Rate", "varname" => "firefoxbd_at_localhost__Ex_Rate", "altvarname" => "Ex_Rate", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Cont", "varname" => "firefoxbd_at_localhost__Exp_Inv_Cont", "altvarname" => "Exp_Inv_Cont", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Head", "varname" => "firefoxbd_at_localhost__Exp_Inv_Head", "altvarname" => "Exp_Inv_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Item", "varname" => "firefoxbd_at_localhost__Exp_Inv_Item", "altvarname" => "Exp_Inv_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Main", "varname" => "firefoxbd_at_localhost__Exp_Inv_Main", "altvarname" => "Exp_Inv_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Pack", "varname" => "firefoxbd_at_localhost__Exp_Inv_Pack", "altvarname" => "Exp_Inv_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Factory", "varname" => "firefoxbd_at_localhost__Factory", "altvarname" => "Factory", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_uggroups", "varname" => "firefoxbd_at_localhost__ffxbderp_uggroups", "altvarname" => "ffxbderp_uggroups", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugmembers", "varname" => "firefoxbd_at_localhost__ffxbderp_ugmembers", "altvarname" => "ffxbderp_ugmembers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugrights", "varname" => "firefoxbd_at_localhost__ffxbderp_ugrights", "altvarname" => "ffxbderp_ugrights", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Fixed_Comments_Shortage", "varname" => "firefoxbd_at_localhost__Fixed_Comments_Shortage", "altvarname" => "Fixed_Comments_Shortage", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Fixed_Inv_Packing", "varname" => "firefoxbd_at_localhost__Fixed_Inv_Packing", "altvarname" => "Fixed_Inv_Packing", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Fork_Master", "varname" => "firefoxbd_at_localhost__Fork_Master", "altvarname" => "Fork_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Forwaders", "varname" => "firefoxbd_at_localhost__Forwaders", "altvarname" => "Forwaders", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Forwaders_Local", "varname" => "firefoxbd_at_localhost__Forwaders_Local", "altvarname" => "Forwaders_Local", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Frame_Drawing_Master", "varname" => "firefoxbd_at_localhost__Frame_Drawing_Master", "altvarname" => "Frame_Drawing_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Fright", "varname" => "firefoxbd_at_localhost__Fright", "altvarname" => "Fright", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Fty_Ports", "varname" => "firefoxbd_at_localhost__Fty_Ports", "altvarname" => "Fty_Ports", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Gsp_Bom_costing", "varname" => "firefoxbd_at_localhost__Gsp_Bom_costing", "altvarname" => "Gsp_Bom_costing", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_Inv", "varname" => "firefoxbd_at_localhost__Ibws_Head_Inv", "altvarname" => "Ibws_Head_Inv", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Inv_Plus_Minus2", "varname" => "firefoxbd_at_localhost__Ibws_Inv_Plus_Minus2", "altvarname" => "Ibws_Inv_Plus_Minus2", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Inv_Remarks2", "varname" => "firefoxbd_at_localhost__Ibws_Inv_Remarks2", "altvarname" => "Ibws_Inv_Remarks2", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "IM_Master", "varname" => "firefoxbd_at_localhost__IM_Master", "altvarname" => "IM_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Imp_F_Remarks", "varname" => "firefoxbd_at_localhost__Imp_F_Remarks", "altvarname" => "Imp_F_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Item_Unit", "varname" => "firefoxbd_at_localhost__Item_Unit", "altvarname" => "Item_Unit", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Loading_Ports", "varname" => "firefoxbd_at_localhost__Loading_Ports", "altvarname" => "Loading_Ports", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Maker", "varname" => "firefoxbd_at_localhost__Maker", "altvarname" => "Maker", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Brake", "varname" => "firefoxbd_at_localhost__Model_Brake", "altvarname" => "Model_Brake", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Decal", "varname" => "firefoxbd_at_localhost__Model_Decal", "altvarname" => "Model_Decal", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_En", "varname" => "firefoxbd_at_localhost__Model_En", "altvarname" => "Model_En", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Fork", "varname" => "firefoxbd_at_localhost__Model_Fork", "altvarname" => "Model_Fork", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Frame", "varname" => "firefoxbd_at_localhost__Model_Frame", "altvarname" => "Model_Frame", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Gear", "varname" => "firefoxbd_at_localhost__Model_Gear", "altvarname" => "Model_Gear", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Group", "varname" => "firefoxbd_at_localhost__Model_Group", "altvarname" => "Model_Group", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Material", "varname" => "firefoxbd_at_localhost__Model_Material", "altvarname" => "Model_Material", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Paint", "varname" => "firefoxbd_at_localhost__Model_Paint", "altvarname" => "Model_Paint", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Pkg", "varname" => "firefoxbd_at_localhost__Model_Pkg", "altvarname" => "Model_Pkg", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Rim", "varname" => "firefoxbd_at_localhost__Model_Rim", "altvarname" => "Model_Rim", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Rimtype", "varname" => "firefoxbd_at_localhost__Model_Rimtype", "altvarname" => "Model_Rimtype", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Shifter", "varname" => "firefoxbd_at_localhost__Model_Shifter", "altvarname" => "Model_Shifter", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Size", "varname" => "firefoxbd_at_localhost__Model_Size", "altvarname" => "Model_Size", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Speed", "varname" => "firefoxbd_at_localhost__Model_Speed", "altvarname" => "Model_Speed", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Spoke", "varname" => "firefoxbd_at_localhost__Model_Spoke", "altvarname" => "Model_Spoke", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Type", "varname" => "firefoxbd_at_localhost__Model_Type", "altvarname" => "Model_Type", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Tyre", "varname" => "firefoxbd_at_localhost__Model_Tyre", "altvarname" => "Model_Tyre", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Model_Valve", "varname" => "firefoxbd_at_localhost__Model_Valve", "altvarname" => "Model_Valve", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Offer_Head", "varname" => "firefoxbd_at_localhost__Offer_Head", "altvarname" => "Offer_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Offer_Item", "varname" => "firefoxbd_at_localhost__Offer_Item", "altvarname" => "Offer_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Packing", "varname" => "firefoxbd_at_localhost__Packing", "altvarname" => "Packing", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Parts", "varname" => "firefoxbd_at_localhost__Parts", "altvarname" => "Parts", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Payterms_new", "varname" => "firefoxbd_at_localhost__Payterms_new", "altvarname" => "Payterms_new", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PC_Head", "varname" => "firefoxbd_at_localhost__PC_Head", "altvarname" => "PC_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PC_Item", "varname" => "firefoxbd_at_localhost__PC_Item", "altvarname" => "PC_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pc_Plus_Minus", "varname" => "firefoxbd_at_localhost__Pc_Plus_Minus", "altvarname" => "Pc_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pc_Remarks", "varname" => "firefoxbd_at_localhost__Pc_Remarks", "altvarname" => "Pc_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pc_Terms", "varname" => "firefoxbd_at_localhost__Pc_Terms", "altvarname" => "Pc_Terms", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_F_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_F_Head", "altvarname" => "PI_Imp_F_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_F_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_F_Item", "altvarname" => "PI_Imp_F_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_F_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_F_Pack", "altvarname" => "PI_Imp_F_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_F_Pack_New", "varname" => "firefoxbd_at_localhost__PI_Imp_F_Pack_New", "altvarname" => "PI_Imp_F_Pack_New", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_F_Plus_Minus", "varname" => "firefoxbd_at_localhost__PI_Imp_F_Plus_Minus", "altvarname" => "PI_Imp_F_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_F_Rmks", "varname" => "firefoxbd_at_localhost__PI_Imp_F_Rmks", "altvarname" => "PI_Imp_F_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_F_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_F_Head", "altvarname" => "PO_Imp_F_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_F_Item", "varname" => "firefoxbd_at_localhost__PO_Imp_F_Item", "altvarname" => "PO_Imp_F_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_F_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Imp_F_Plus_Minus", "altvarname" => "PO_Imp_F_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_F", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_F", "altvarname" => "PO_PSI_Imp_F", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_F_List", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_F_List", "altvarname" => "PO_PSI_Imp_F_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_Head", "varname" => "firefoxbd_at_localhost__PSI_Imp_F_Head", "altvarname" => "PSI_Imp_F_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_Item", "varname" => "firefoxbd_at_localhost__PSI_Imp_F_Item", "altvarname" => "PSI_Imp_F_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_Main", "varname" => "firefoxbd_at_localhost__PSI_Imp_F_Main", "altvarname" => "PSI_Imp_F_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_plusminus", "varname" => "firefoxbd_at_localhost__PSI_Imp_F_plusminus", "altvarname" => "PSI_Imp_F_plusminus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_Remarks", "varname" => "firefoxbd_at_localhost__PSI_Imp_F_Remarks", "altvarname" => "PSI_Imp_F_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Region", "varname" => "firefoxbd_at_localhost__Region", "altvarname" => "Region", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Replacement_Item", "varname" => "firefoxbd_at_localhost__Replacement_Item", "altvarname" => "Replacement_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Head", "varname" => "firefoxbd_at_localhost__Sc_Head", "altvarname" => "Sc_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SC_Item", "varname" => "firefoxbd_at_localhost__SC_Item", "altvarname" => "SC_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Item_FW", "varname" => "firefoxbd_at_localhost__Sc_Item_FW", "altvarname" => "Sc_Item_FW", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Item_RW", "varname" => "firefoxbd_at_localhost__Sc_Item_RW", "altvarname" => "Sc_Item_RW", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Itemnot_Include", "varname" => "firefoxbd_at_localhost__Sc_Itemnot_Include", "altvarname" => "Sc_Itemnot_Include", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SC_Options", "varname" => "firefoxbd_at_localhost__SC_Options", "altvarname" => "SC_Options", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Plus_Minus", "varname" => "firefoxbd_at_localhost__Sc_Plus_Minus", "altvarname" => "Sc_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Remarks", "varname" => "firefoxbd_at_localhost__Sc_Remarks", "altvarname" => "Sc_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Reusable", "varname" => "firefoxbd_at_localhost__Sc_Reusable", "altvarname" => "Sc_Reusable", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SC_Spec_Comp", "varname" => "firefoxbd_at_localhost__SC_Spec_Comp", "altvarname" => "SC_Spec_Comp", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Shipping_Marks", "varname" => "firefoxbd_at_localhost__Shipping_Marks", "altvarname" => "Shipping_Marks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sp_Kitem", "varname" => "firefoxbd_at_localhost__Sp_Kitem", "altvarname" => "Sp_Kitem", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Amendments", "varname" => "firefoxbd_at_localhost__Spec_Amendments", "altvarname" => "Spec_Amendments", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Amendments_Item", "varname" => "firefoxbd_at_localhost__Spec_Amendments_Item", "altvarname" => "Spec_Amendments_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Group", "varname" => "firefoxbd_at_localhost__Spec_Group", "altvarname" => "Spec_Group", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Group_Item", "varname" => "firefoxbd_at_localhost__Spec_Group_Item", "altvarname" => "Spec_Group_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_History", "varname" => "firefoxbd_at_localhost__Spec_History", "altvarname" => "Spec_History", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Item_Color", "varname" => "firefoxbd_at_localhost__Spec_Item_Color", "altvarname" => "Spec_Item_Color", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Item_Frsize", "varname" => "firefoxbd_at_localhost__Spec_Item_Frsize", "altvarname" => "Spec_Item_Frsize", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Kids_Head", "varname" => "firefoxbd_at_localhost__Spec_Kids_Head", "altvarname" => "Spec_Kids_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Material", "varname" => "firefoxbd_at_localhost__Spec_Material", "altvarname" => "Spec_Material", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Price_List", "varname" => "firefoxbd_at_localhost__Spec_Price_List", "altvarname" => "Spec_Price_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Type", "varname" => "firefoxbd_at_localhost__Spec_Type", "altvarname" => "Spec_Type", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Fixed", "varname" => "firefoxbd_at_localhost__ST_Item_Fixed", "altvarname" => "ST_Item_Fixed", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Supplier", "varname" => "firefoxbd_at_localhost__Supplier", "altvarname" => "Supplier", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "TC_Modelwise", "varname" => "firefoxbd_at_localhost__TC_Modelwise", "altvarname" => "TC_Modelwise", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Term_From_After", "varname" => "firefoxbd_at_localhost__Term_From_After", "altvarname" => "Term_From_After", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Title", "varname" => "firefoxbd_at_localhost__Title", "altvarname" => "Title", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "User_Rights", "varname" => "firefoxbd_at_localhost__User_Rights", "altvarname" => "User_Rights", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "webusers", "varname" => "firefoxbd_at_localhost__webusers", "altvarname" => "webusers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Welding_Status", "varname" => "firefoxbd_at_localhost__Welding_Status", "altvarname" => "Welding_Status", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Weldingstatus", "varname" => "firefoxbd_at_localhost__Weldingstatus", "altvarname" => "Weldingstatus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>