<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_103_6_168_141" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "Assets_Group", "varname" => "firefoxbd_at_103_6_168_141__Assets_Group", "altvarname" => "Assets_Group", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Assets_Item", "varname" => "firefoxbd_at_103_6_168_141__Assets_Item", "altvarname" => "Assets_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_audit", "varname" => "firefoxbd_at_103_6_168_141__BD_audit", "altvarname" => "BD_audit", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Model", "varname" => "firefoxbd_at_103_6_168_141__BD_Model", "altvarname" => "BD_Model", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Component_QC_Head", "varname" => "firefoxbd_at_103_6_168_141__Component_QC_Head", "altvarname" => "Component_QC_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Component_QC_Item", "varname" => "firefoxbd_at_103_6_168_141__Component_QC_Item", "altvarname" => "Component_QC_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "CusDec", "varname" => "firefoxbd_at_103_6_168_141__CusDec", "altvarname" => "CusDec", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_uggroups", "varname" => "firefoxbd_at_103_6_168_141__ffxbderp_uggroups", "altvarname" => "ffxbderp_uggroups", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugmembers", "varname" => "firefoxbd_at_103_6_168_141__ffxbderp_ugmembers", "altvarname" => "ffxbderp_ugmembers", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugrights", "varname" => "firefoxbd_at_103_6_168_141__ffxbderp_ugrights", "altvarname" => "ffxbderp_ugrights", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Parts", "varname" => "firefoxbd_at_103_6_168_141__Parts", "altvarname" => "Parts", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_A_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_A_Head", "altvarname" => "PI_Imp_A_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_C_Head", "altvarname" => "PI_Imp_C_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Paint_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_Paint_Head", "altvarname" => "PI_Imp_Paint_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_R_Head", "altvarname" => "PI_Imp_R_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_A_Head", "varname" => "firefoxbd_at_103_6_168_141__PO_Loc_A_Head", "altvarname" => "PO_Loc_A_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_B_Head", "varname" => "firefoxbd_at_103_6_168_141__PO_Loc_B_Head", "altvarname" => "PO_Loc_B_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_C_Head", "varname" => "firefoxbd_at_103_6_168_141__PO_Loc_C_Head", "altvarname" => "PO_Loc_C_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_F_Head", "varname" => "firefoxbd_at_103_6_168_141__PO_Loc_F_Head", "altvarname" => "PO_Loc_F_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ref_Fixed_Inv1", "varname" => "firefoxbd_at_103_6_168_141__Ref_Fixed_Inv1", "altvarname" => "Ref_Fixed_Inv1", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Replacement_Item", "varname" => "firefoxbd_at_103_6_168_141__Replacement_Item", "altvarname" => "Replacement_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Request_Imp_Permit_Head", "varname" => "firefoxbd_at_103_6_168_141__Request_Imp_Permit_Head", "altvarname" => "Request_Imp_Permit_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Request_Imp_Permit_Item", "varname" => "firefoxbd_at_103_6_168_141__Request_Imp_Permit_Item", "altvarname" => "Request_Imp_Permit_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Rim_DeliveryNote_Head", "varname" => "firefoxbd_at_103_6_168_141__Rim_DeliveryNote_Head", "altvarname" => "Rim_DeliveryNote_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Head", "varname" => "firefoxbd_at_103_6_168_141__Sc_Head", "altvarname" => "Sc_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SC_Item", "varname" => "firefoxbd_at_103_6_168_141__SC_Item", "altvarname" => "SC_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Itemnot_Include", "varname" => "firefoxbd_at_103_6_168_141__Sc_Itemnot_Include", "altvarname" => "Sc_Itemnot_Include", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Group", "varname" => "firefoxbd_at_103_6_168_141__Spec_Group", "altvarname" => "Spec_Group", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Adj_Reason", "varname" => "firefoxbd_at_103_6_168_141__ST_Adj_Reason", "altvarname" => "ST_Adj_Reason", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Adj_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_B_Adj_Item", "altvarname" => "ST_B_Adj_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_B_Isu_Head", "altvarname" => "ST_B_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_B_Isu_Item", "altvarname" => "ST_B_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Rec_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_B_Rec_Head", "altvarname" => "ST_B_Rec_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Rec_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_B_Rec_Item", "altvarname" => "ST_B_Rec_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Ret_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_B_Ret_Head", "altvarname" => "ST_B_Ret_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Ret_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_B_Ret_Item", "altvarname" => "ST_B_Ret_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Adj_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Adj_Item", "altvarname" => "ST_C_Adj_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Isu_Head", "altvarname" => "ST_C_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Head_New", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Isu_Head_New", "altvarname" => "ST_C_Isu_Head_New", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Isu_Item", "altvarname" => "ST_C_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Item_New", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Isu_Item_New", "altvarname" => "ST_C_Isu_Item_New", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Rec_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Rec_Head", "altvarname" => "ST_C_Rec_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Rec_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Rec_Item", "altvarname" => "ST_C_Rec_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Ret_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Ret_Head", "altvarname" => "ST_C_Ret_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Ret_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_C_Ret_Item", "altvarname" => "ST_C_Ret_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Cap_Adj_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Cap_Adj_Item", "altvarname" => "ST_Cap_Adj_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Cap_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Cap_Isu_Head", "altvarname" => "ST_Cap_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Cap_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Cap_Isu_Item", "altvarname" => "ST_Cap_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Cap_Rec_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Cap_Rec_Head", "altvarname" => "ST_Cap_Rec_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Cap_Rec_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Cap_Rec_Item", "altvarname" => "ST_Cap_Rec_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Cap_Ret_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Cap_Ret_Head", "altvarname" => "ST_Cap_Ret_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Cap_Ret_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Cap_Ret_Item", "altvarname" => "ST_Cap_Ret_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Department", "varname" => "firefoxbd_at_103_6_168_141__ST_Department", "altvarname" => "ST_Department", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Adj_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_F_Adj_Item", "altvarname" => "ST_F_Adj_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_F_Isu_Head", "altvarname" => "ST_F_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_F_Isu_Item", "altvarname" => "ST_F_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Rec_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_F_Rec_Head", "altvarname" => "ST_F_Rec_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Rec_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_F_Rec_Item", "altvarname" => "ST_F_Rec_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Ret_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_F_Ret_Head", "altvarname" => "ST_F_Ret_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Ret_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_F_Ret_Item", "altvarname" => "ST_F_Ret_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Fixed_AdjLed_Rec_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Fixed_AdjLed_Rec_Head", "altvarname" => "ST_Fixed_AdjLed_Rec_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Adj_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Adj_Item", "altvarname" => "ST_IBWS_Adj_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Isu_Exframe_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Isu_Exframe_Head", "altvarname" => "ST_IBWS_Isu_Exframe_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Isu_Exframe_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Isu_Exframe_Item", "altvarname" => "ST_IBWS_Isu_Exframe_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Isu_Head", "altvarname" => "ST_IBWS_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Isu_Head_Sp", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Isu_Head_Sp", "altvarname" => "ST_IBWS_Isu_Head_Sp", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Isu_Item", "altvarname" => "ST_IBWS_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Isu_Item_Sp", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Isu_Item_Sp", "altvarname" => "ST_IBWS_Isu_Item_Sp", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_PC3_Rec_Head_New", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_PC3_Rec_Head_New", "altvarname" => "ST_IBWS_PC3_Rec_Head_New", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_PC3_Rec_Item_New", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_PC3_Rec_Item_New", "altvarname" => "ST_IBWS_PC3_Rec_Item_New", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Rec_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Rec_Head", "altvarname" => "ST_IBWS_Rec_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Rec_Head_New", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Rec_Head_New", "altvarname" => "ST_IBWS_Rec_Head_New", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Rec_Head_Sp", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Rec_Head_Sp", "altvarname" => "ST_IBWS_Rec_Head_Sp", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Rec_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Rec_Item", "altvarname" => "ST_IBWS_Rec_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Rec_Item_New", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Rec_Item_New", "altvarname" => "ST_IBWS_Rec_Item_New", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Rec_Item_Sp", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Rec_Item_Sp", "altvarname" => "ST_IBWS_Rec_Item_Sp", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Ret_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Ret_Head", "altvarname" => "ST_IBWS_Ret_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Ret_Head_Sp", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Ret_Head_Sp", "altvarname" => "ST_IBWS_Ret_Head_Sp", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Ret_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Ret_Item", "altvarname" => "ST_IBWS_Ret_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_IBWS_Ret_Item_Sp", "varname" => "firefoxbd_at_103_6_168_141__ST_IBWS_Ret_Item_Sp", "altvarname" => "ST_IBWS_Ret_Item_Sp", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Item", "altvarname" => "ST_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Bulk", "varname" => "firefoxbd_at_103_6_168_141__ST_Item_Bulk", "altvarname" => "ST_Item_Bulk", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Fixed", "varname" => "firefoxbd_at_103_6_168_141__ST_Item_Fixed", "altvarname" => "ST_Item_Fixed", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Location", "varname" => "firefoxbd_at_103_6_168_141__ST_Item_Location", "altvarname" => "ST_Item_Location", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Paint_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Paint_Isu_Head", "altvarname" => "ST_Paint_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Paint_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Paint_Isu_Item", "altvarname" => "ST_Paint_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Rep_Rec_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Rep_Rec_Item", "altvarname" => "ST_Rep_Rec_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Rep_Ret_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Rep_Ret_Head", "altvarname" => "ST_Rep_Ret_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Repl_Adj_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Repl_Adj_Item", "altvarname" => "ST_Repl_Adj_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Repl_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Repl_Isu_Head", "altvarname" => "ST_Repl_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Repl_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Repl_Isu_Item", "altvarname" => "ST_Repl_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Repl_Rec_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_Repl_Rec_Head", "altvarname" => "ST_Repl_Rec_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Repl_Ret_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_Repl_Ret_Item", "altvarname" => "ST_Repl_Ret_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_ReplFrm_Isu_Head", "varname" => "firefoxbd_at_103_6_168_141__ST_ReplFrm_Isu_Head", "altvarname" => "ST_ReplFrm_Isu_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_ReplFrm_Isu_Item", "varname" => "firefoxbd_at_103_6_168_141__ST_ReplFrm_Isu_Item", "altvarname" => "ST_ReplFrm_Isu_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Swap_Fixed", "varname" => "firefoxbd_at_103_6_168_141__ST_Swap_Fixed", "altvarname" => "ST_Swap_Fixed", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ST_Swap_IBWS", "varname" => "firefoxbd_at_103_6_168_141__ST_Swap_IBWS", "altvarname" => "ST_Swap_IBWS", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "TRI_Exp_Inv_Cont", "varname" => "firefoxbd_at_103_6_168_141__TRI_Exp_Inv_Cont", "altvarname" => "TRI_Exp_Inv_Cont", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "TRI_Exp_Inv_Main", "varname" => "firefoxbd_at_103_6_168_141__TRI_Exp_Inv_Main", "altvarname" => "TRI_Exp_Inv_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "webusers", "varname" => "firefoxbd_at_103_6_168_141__webusers", "altvarname" => "webusers", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>