<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_103_6_168_141" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "Acc_DrCr_Base", "varname" => "firefoxbd_at_103_6_168_141__Acc_DrCr_Base", "altvarname" => "Acc_DrCr_Base", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Acc_DrCr_Credit_Head", "varname" => "firefoxbd_at_103_6_168_141__Acc_DrCr_Credit_Head", "altvarname" => "Acc_DrCr_Credit_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Acc_DrCr_Credit_Item", "varname" => "firefoxbd_at_103_6_168_141__Acc_DrCr_Credit_Item", "altvarname" => "Acc_DrCr_Credit_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Acc_DrCr_Debit_Head", "varname" => "firefoxbd_at_103_6_168_141__Acc_DrCr_Debit_Head", "altvarname" => "Acc_DrCr_Debit_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Acc_DrCr_Debit_Item", "varname" => "firefoxbd_at_103_6_168_141__Acc_DrCr_Debit_Item", "altvarname" => "Acc_DrCr_Debit_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Agent_Master", "varname" => "firefoxbd_at_103_6_168_141__Agent_Master", "altvarname" => "Agent_Master", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_audit", "varname" => "firefoxbd_at_103_6_168_141__BD_audit", "altvarname" => "BD_audit", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Reimbursement", "varname" => "firefoxbd_at_103_6_168_141__BD_Reimbursement", "altvarname" => "BD_Reimbursement", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_Banks", "varname" => "firefoxbd_at_103_6_168_141__Cash_Flow_Banks", "altvarname" => "Cash_Flow_Banks", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_CashInHand", "varname" => "firefoxbd_at_103_6_168_141__Cash_Flow_CashInHand", "altvarname" => "Cash_Flow_CashInHand", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_CashInHand_Item", "varname" => "firefoxbd_at_103_6_168_141__Cash_Flow_CashInHand_Item", "altvarname" => "Cash_Flow_CashInHand_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_IncomeOthers", "varname" => "firefoxbd_at_103_6_168_141__Cash_Flow_IncomeOthers", "altvarname" => "Cash_Flow_IncomeOthers", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_Loan", "varname" => "firefoxbd_at_103_6_168_141__Cash_Flow_Loan", "altvarname" => "Cash_Flow_Loan", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_OthersExpence", "varname" => "firefoxbd_at_103_6_168_141__Cash_Flow_OthersExpence", "altvarname" => "Cash_Flow_OthersExpence", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cash_Flow_Payments_Expences", "varname" => "firefoxbd_at_103_6_168_141__Cash_Flow_Payments_Expences", "altvarname" => "Cash_Flow_Payments_Expences", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Combine_FTY_Supplier", "varname" => "firefoxbd_at_103_6_168_141__Combine_FTY_Supplier", "altvarname" => "Combine_FTY_Supplier", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Currency", "varname" => "firefoxbd_at_103_6_168_141__Currency", "altvarname" => "Currency", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer", "varname" => "firefoxbd_at_103_6_168_141__Customer", "altvarname" => "Customer", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Factory", "varname" => "firefoxbd_at_103_6_168_141__Factory", "altvarname" => "Factory", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_uggroups", "varname" => "firefoxbd_at_103_6_168_141__ffxbderp_uggroups", "altvarname" => "ffxbderp_uggroups", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugmembers", "varname" => "firefoxbd_at_103_6_168_141__ffxbderp_ugmembers", "altvarname" => "ffxbderp_ugmembers", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugrights", "varname" => "firefoxbd_at_103_6_168_141__ffxbderp_ugrights", "altvarname" => "ffxbderp_ugrights", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Freight_IBWS", "varname" => "firefoxbd_at_103_6_168_141__Freight_IBWS", "altvarname" => "Freight_IBWS", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "IBWS_Bcharge", "varname" => "firefoxbd_at_103_6_168_141__IBWS_Bcharge", "altvarname" => "IBWS_Bcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "IBWS_Inv_SCwise", "varname" => "firefoxbd_at_103_6_168_141__IBWS_Inv_SCwise", "altvarname" => "IBWS_Inv_SCwise", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "IBWS_Inv_SCwisespttl", "varname" => "firefoxbd_at_103_6_168_141__IBWS_Inv_SCwisespttl", "altvarname" => "IBWS_Inv_SCwisespttl", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "IBWS_Payment_Main", "varname" => "firefoxbd_at_103_6_168_141__IBWS_Payment_Main", "altvarname" => "IBWS_Payment_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Merchandiser", "varname" => "firefoxbd_at_103_6_168_141__Merchandiser", "altvarname" => "Merchandiser", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_A_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_A_Head", "altvarname" => "PI_Imp_A_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_B_Head", "altvarname" => "PI_Imp_B_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_C_Head", "altvarname" => "PI_Imp_C_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_F_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_F_Head", "altvarname" => "PI_Imp_F_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Paint_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_Paint_Head", "altvarname" => "PI_Imp_Paint_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Head", "varname" => "firefoxbd_at_103_6_168_141__PI_Imp_R_Head", "altvarname" => "PI_Imp_R_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Bulk_Bankcharge", "varname" => "firefoxbd_at_103_6_168_141__Po_Bulk_Bankcharge", "altvarname" => "Po_Bulk_Bankcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Bulk_Freight", "varname" => "firefoxbd_at_103_6_168_141__Po_Bulk_Freight", "altvarname" => "Po_Bulk_Freight", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Bulk_Pay", "varname" => "firefoxbd_at_103_6_168_141__Po_Bulk_Pay", "altvarname" => "Po_Bulk_Pay", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Cap_Bankcharge", "varname" => "firefoxbd_at_103_6_168_141__Po_Cap_Bankcharge", "altvarname" => "Po_Cap_Bankcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Cap_Freight", "varname" => "firefoxbd_at_103_6_168_141__Po_Cap_Freight", "altvarname" => "Po_Cap_Freight", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Cap_Pay", "varname" => "firefoxbd_at_103_6_168_141__Po_Cap_Pay", "altvarname" => "Po_Cap_Pay", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Con_Bankcharge", "varname" => "firefoxbd_at_103_6_168_141__Po_Con_Bankcharge", "altvarname" => "Po_Con_Bankcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Con_Freight", "varname" => "firefoxbd_at_103_6_168_141__Po_Con_Freight", "altvarname" => "Po_Con_Freight", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Con_Pay", "varname" => "firefoxbd_at_103_6_168_141__Po_Con_Pay", "altvarname" => "Po_Con_Pay", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Fixed_Bankcharge", "varname" => "firefoxbd_at_103_6_168_141__Po_Fixed_Bankcharge", "altvarname" => "Po_Fixed_Bankcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Fixed_Freight", "varname" => "firefoxbd_at_103_6_168_141__Po_Fixed_Freight", "altvarname" => "Po_Fixed_Freight", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Fixed_Pay", "varname" => "firefoxbd_at_103_6_168_141__Po_Fixed_Pay", "altvarname" => "Po_Fixed_Pay", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Paint_Bankcharge", "varname" => "firefoxbd_at_103_6_168_141__Po_Paint_Bankcharge", "altvarname" => "Po_Paint_Bankcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Paint_Freight", "varname" => "firefoxbd_at_103_6_168_141__Po_Paint_Freight", "altvarname" => "Po_Paint_Freight", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Paint_Pay", "varname" => "firefoxbd_at_103_6_168_141__Po_Paint_Pay", "altvarname" => "Po_Paint_Pay", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Rc_Bankcharge", "varname" => "firefoxbd_at_103_6_168_141__Po_Rc_Bankcharge", "altvarname" => "Po_Rc_Bankcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Rc_Freight", "varname" => "firefoxbd_at_103_6_168_141__Po_Rc_Freight", "altvarname" => "Po_Rc_Freight", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Rc_Pay", "varname" => "firefoxbd_at_103_6_168_141__Po_Rc_Pay", "altvarname" => "Po_Rc_Pay", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Rep_Bankcharge", "varname" => "firefoxbd_at_103_6_168_141__Po_Rep_Bankcharge", "altvarname" => "Po_Rep_Bankcharge", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Repl_Freight", "varname" => "firefoxbd_at_103_6_168_141__Po_Repl_Freight", "altvarname" => "Po_Repl_Freight", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Po_Repl_Pay", "varname" => "firefoxbd_at_103_6_168_141__Po_Repl_Pay", "altvarname" => "Po_Repl_Pay", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_Head", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_A_Head", "altvarname" => "PSI_Imp_A_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_Item", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_A_Item", "altvarname" => "PSI_Imp_A_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_Main", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_A_Main", "altvarname" => "PSI_Imp_A_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Head", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_B_Head", "altvarname" => "PSI_Imp_B_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Item", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_B_Item", "altvarname" => "PSI_Imp_B_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Main", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_B_Main", "altvarname" => "PSI_Imp_B_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Head", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_C_Head", "altvarname" => "PSI_Imp_C_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Item", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_C_Item", "altvarname" => "PSI_Imp_C_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Main", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_C_Main", "altvarname" => "PSI_Imp_C_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_Head", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_F_Head", "altvarname" => "PSI_Imp_F_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_Item", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_F_Item", "altvarname" => "PSI_Imp_F_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_F_Main", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_F_Main", "altvarname" => "PSI_Imp_F_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_Head", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_Paint_Head", "altvarname" => "PSI_Imp_Paint_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_Item", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_Paint_Item", "altvarname" => "PSI_Imp_Paint_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_Main", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_Paint_Main", "altvarname" => "PSI_Imp_Paint_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Head", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_R_Head", "altvarname" => "PSI_Imp_R_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Item", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_R_Item", "altvarname" => "PSI_Imp_R_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Main", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_R_Main", "altvarname" => "PSI_Imp_R_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_Head", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_Rc_Head", "altvarname" => "PSI_Imp_Rc_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_Item", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_Rc_Item", "altvarname" => "PSI_Imp_Rc_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_Main", "varname" => "firefoxbd_at_103_6_168_141__PSI_Imp_Rc_Main", "altvarname" => "PSI_Imp_Rc_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Supplier", "varname" => "firefoxbd_at_103_6_168_141__Supplier", "altvarname" => "Supplier", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Title", "varname" => "firefoxbd_at_103_6_168_141__Title", "altvarname" => "Title", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Utility_Budget_Head_TW", "varname" => "firefoxbd_at_103_6_168_141__Utility_Budget_Head_TW", "altvarname" => "Utility_Budget_Head_TW", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Utility_Budget_Item_TW", "varname" => "firefoxbd_at_103_6_168_141__Utility_Budget_Item_TW", "altvarname" => "Utility_Budget_Item_TW", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Utility_Category", "varname" => "firefoxbd_at_103_6_168_141__Utility_Category", "altvarname" => "Utility_Category", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Utility_Category_TW", "varname" => "firefoxbd_at_103_6_168_141__Utility_Category_TW", "altvarname" => "Utility_Category_TW", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Utility_Head_TW", "varname" => "firefoxbd_at_103_6_168_141__Utility_Head_TW", "altvarname" => "Utility_Head_TW", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Utility_SubCategory", "varname" => "firefoxbd_at_103_6_168_141__Utility_SubCategory", "altvarname" => "Utility_SubCategory", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Utility_SubCategory_TW", "varname" => "firefoxbd_at_103_6_168_141__Utility_SubCategory_TW", "altvarname" => "Utility_SubCategory_TW", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "webusers", "varname" => "firefoxbd_at_103_6_168_141__webusers", "altvarname" => "webusers", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>