<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_103_6_168_141" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "BD_audit", "varname" => "firefoxbd_at_103_6_168_141__BD_audit", "altvarname" => "BD_audit", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "BD_Model", "varname" => "firefoxbd_at_103_6_168_141__BD_Model", "altvarname" => "BD_Model", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Currency", "varname" => "firefoxbd_at_103_6_168_141__Currency", "altvarname" => "Currency", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cus_Ref_Cmt_Head", "varname" => "firefoxbd_at_103_6_168_141__Cus_Ref_Cmt_Head", "altvarname" => "Cus_Ref_Cmt_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Cus_Ref_Cmt_Item", "varname" => "firefoxbd_at_103_6_168_141__Cus_Ref_Cmt_Item", "altvarname" => "Cus_Ref_Cmt_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer", "varname" => "firefoxbd_at_103_6_168_141__Customer", "altvarname" => "Customer", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer_Create_SC", "varname" => "firefoxbd_at_103_6_168_141__Customer_Create_SC", "altvarname" => "Customer_Create_SC", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer_Create_SC_Item", "varname" => "firefoxbd_at_103_6_168_141__Customer_Create_SC_Item", "altvarname" => "Customer_Create_SC_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer_Create_SC_Rmks", "varname" => "firefoxbd_at_103_6_168_141__Customer_Create_SC_Rmks", "altvarname" => "Customer_Create_SC_Rmks", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Customer_Login", "varname" => "firefoxbd_at_103_6_168_141__Customer_Login", "altvarname" => "Customer_Login", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Destinations", "varname" => "firefoxbd_at_103_6_168_141__Destinations", "altvarname" => "Destinations", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ex_Rate", "varname" => "firefoxbd_at_103_6_168_141__Ex_Rate", "altvarname" => "Ex_Rate", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Cont", "varname" => "firefoxbd_at_103_6_168_141__Exp_Inv_Cont", "altvarname" => "Exp_Inv_Cont", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Main", "varname" => "firefoxbd_at_103_6_168_141__Exp_Inv_Main", "altvarname" => "Exp_Inv_Main", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Fright", "varname" => "firefoxbd_at_103_6_168_141__Fright", "altvarname" => "Fright", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Gatepass_Head", "varname" => "firefoxbd_at_103_6_168_141__Gatepass_Head", "altvarname" => "Gatepass_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Invoice_Head", "varname" => "firefoxbd_at_103_6_168_141__Invoice_Head", "altvarname" => "Invoice_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Invoice_item", "varname" => "firefoxbd_at_103_6_168_141__Invoice_item", "altvarname" => "Invoice_item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Item_Unit", "varname" => "firefoxbd_at_103_6_168_141__Item_Unit", "altvarname" => "Item_Unit", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Loading_Ports", "varname" => "firefoxbd_at_103_6_168_141__Loading_Ports", "altvarname" => "Loading_Ports", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Maker", "varname" => "firefoxbd_at_103_6_168_141__Maker", "altvarname" => "Maker", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Offer_Head", "varname" => "firefoxbd_at_103_6_168_141__Offer_Head", "altvarname" => "Offer_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Packing", "varname" => "firefoxbd_at_103_6_168_141__Packing", "altvarname" => "Packing", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Payterms_new", "varname" => "firefoxbd_at_103_6_168_141__Payterms_new", "altvarname" => "Payterms_new", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Pc_Terms", "varname" => "firefoxbd_at_103_6_168_141__Pc_Terms", "altvarname" => "Pc_Terms", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ref_ITLinv", "varname" => "firefoxbd_at_103_6_168_141__Ref_ITLinv", "altvarname" => "Ref_ITLinv", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Ref_Spec_amend", "varname" => "firefoxbd_at_103_6_168_141__Ref_Spec_amend", "altvarname" => "Ref_Spec_amend", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Sc_Head", "varname" => "firefoxbd_at_103_6_168_141__Sc_Head", "altvarname" => "Sc_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SC_Item", "varname" => "firefoxbd_at_103_6_168_141__SC_Item", "altvarname" => "SC_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SHP_Adv_Data", "varname" => "firefoxbd_at_103_6_168_141__SHP_Adv_Data", "altvarname" => "SHP_Adv_Data", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "SHP_Adv_Head", "varname" => "firefoxbd_at_103_6_168_141__SHP_Adv_Head", "altvarname" => "SHP_Adv_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Amendments", "varname" => "firefoxbd_at_103_6_168_141__Spec_Amendments", "altvarname" => "Spec_Amendments", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Amendments_Item", "varname" => "firefoxbd_at_103_6_168_141__Spec_Amendments_Item", "altvarname" => "Spec_Amendments_Item", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Group", "varname" => "firefoxbd_at_103_6_168_141__Spec_Group", "altvarname" => "Spec_Group", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_History", "varname" => "firefoxbd_at_103_6_168_141__Spec_History", "altvarname" => "Spec_History", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Item_Color", "varname" => "firefoxbd_at_103_6_168_141__Spec_Item_Color", "altvarname" => "Spec_Item_Color", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Kids_Head", "varname" => "firefoxbd_at_103_6_168_141__Spec_Kids_Head", "altvarname" => "Spec_Kids_Head", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_Type", "varname" => "firefoxbd_at_103_6_168_141__Spec_Type", "altvarname" => "Spec_Type", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Spec_withmodel", "varname" => "firefoxbd_at_103_6_168_141__Spec_withmodel", "altvarname" => "Spec_withmodel", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Specrevision", "varname" => "firefoxbd_at_103_6_168_141__Specrevision", "altvarname" => "Specrevision", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Upload_Export", "varname" => "firefoxbd_at_103_6_168_141__Upload_Export", "altvarname" => "Upload_Export", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Upload_PPsample", "varname" => "firefoxbd_at_103_6_168_141__Upload_PPsample", "altvarname" => "Upload_PPsample", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Upload_QC", "varname" => "firefoxbd_at_103_6_168_141__Upload_QC", "altvarname" => "Upload_QC", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Upload_Sc_Att", "varname" => "firefoxbd_at_103_6_168_141__Upload_Sc_Att", "altvarname" => "Upload_Sc_Att", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
			$dalTables[$conn->connId][] = array("name" => "Upload_Spec_Comp", "varname" => "firefoxbd_at_103_6_168_141__Upload_Spec_Comp", "altvarname" => "Upload_Spec_Comp", "connId" => "firefoxbd_at_103_6_168_141", "schema" => "", "connName" => "firefoxbd at 103.6.168.141");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>