<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_erp_corvoltd_com" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "Agent_Master", "varname" => "firefoxbd_at_erp_corvoltd_com__Agent_Master", "altvarname" => "Agent_Master", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "BD_Model", "varname" => "firefoxbd_at_erp_corvoltd_com__BD_Model", "altvarname" => "BD_Model", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Cost_Main", "varname" => "firefoxbd_at_erp_corvoltd_com__Cost_Main", "altvarname" => "Cost_Main", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "CusDec", "varname" => "firefoxbd_at_erp_corvoltd_com__CusDec", "altvarname" => "CusDec", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "CusDec_BD", "varname" => "firefoxbd_at_erp_corvoltd_com__CusDec_BD", "altvarname" => "CusDec_BD", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "CusDec_BD_PC3", "varname" => "firefoxbd_at_erp_corvoltd_com__CusDec_BD_PC3", "altvarname" => "CusDec_BD_PC3", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "CusDec_Pclist", "varname" => "firefoxbd_at_erp_corvoltd_com__CusDec_Pclist", "altvarname" => "CusDec_Pclist", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Cont", "varname" => "firefoxbd_at_erp_corvoltd_com__Exp_Inv_Cont", "altvarname" => "Exp_Inv_Cont", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__Exp_Inv_Head", "altvarname" => "Exp_Inv_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__Exp_Inv_Item", "altvarname" => "Exp_Inv_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Exp_Inv_Main", "varname" => "firefoxbd_at_erp_corvoltd_com__Exp_Inv_Main", "altvarname" => "Exp_Inv_Main", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "FGS_Actual_Receipt", "varname" => "firefoxbd_at_erp_corvoltd_com__FGS_Actual_Receipt", "altvarname" => "FGS_Actual_Receipt", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "FrameComponents_SCwise", "varname" => "firefoxbd_at_erp_corvoltd_com__FrameComponents_SCwise", "altvarname" => "FrameComponents_SCwise", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "gsp_analysis", "varname" => "firefoxbd_at_erp_corvoltd_com__gsp_analysis", "altvarname" => "gsp_analysis", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "gsp_analysis_sc_seq", "varname" => "firefoxbd_at_erp_corvoltd_com__gsp_analysis_sc_seq", "altvarname" => "gsp_analysis_sc_seq", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Gsp_Bom_costing", "varname" => "firefoxbd_at_erp_corvoltd_com__Gsp_Bom_costing", "altvarname" => "Gsp_Bom_costing", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data", "altvarname" => "Ibws_Data", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data_CH", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data_CH", "altvarname" => "Ibws_Data_CH", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data_shp_qty", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data_shp_qty", "altvarname" => "Ibws_Data_shp_qty", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data_shp_qty_04092020", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data_shp_qty_04092020", "altvarname" => "Ibws_Data_shp_qty_04092020", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data_shp_qty_pc3", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data_shp_qty_pc3", "altvarname" => "Ibws_Data_shp_qty_pc3", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data_shp_qty_pc3_07092020", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data_shp_qty_pc3_07092020", "altvarname" => "Ibws_Data_shp_qty_pc3_07092020", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data_SP_PC3", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data_SP_PC3", "altvarname" => "Ibws_Data_SP_PC3", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Data_Status", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Data_Status", "altvarname" => "Ibws_Data_Status", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Head", "altvarname" => "Ibws_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_CH", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Head_CH", "altvarname" => "Ibws_Head_CH", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_Inv", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Head_Inv", "altvarname" => "Ibws_Head_Inv", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_Inv_PC3", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Head_Inv_PC3", "altvarname" => "Ibws_Head_Inv_PC3", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_Inv_PcList", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Head_Inv_PcList", "altvarname" => "Ibws_Head_Inv_PcList", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_Inv_PcList_PC3", "varname" => "firefoxbd_at_erp_corvoltd_com__Ibws_Head_Inv_PcList_PC3", "altvarname" => "Ibws_Head_Inv_PcList_PC3", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Invoice_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__Invoice_Head", "altvarname" => "Invoice_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Invoice_item", "varname" => "firefoxbd_at_erp_corvoltd_com__Invoice_item", "altvarname" => "Invoice_item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Invoice_Pack", "varname" => "firefoxbd_at_erp_corvoltd_com__Invoice_Pack", "altvarname" => "Invoice_Pack", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Loading_Head_New", "varname" => "firefoxbd_at_erp_corvoltd_com__Loading_Head_New", "altvarname" => "Loading_Head_New", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Loading_Item_New", "varname" => "firefoxbd_at_erp_corvoltd_com__Loading_Item_New", "altvarname" => "Loading_Item_New", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Paint_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__Paint_Head", "altvarname" => "Paint_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Paint_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__Paint_Item", "altvarname" => "Paint_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Paint_Item_Final", "varname" => "firefoxbd_at_erp_corvoltd_com__Paint_Item_Final", "altvarname" => "Paint_Item_Final", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Paint_Item_Gin", "varname" => "firefoxbd_at_erp_corvoltd_com__Paint_Item_Gin", "altvarname" => "Paint_Item_Gin", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Paint_Item_Request", "varname" => "firefoxbd_at_erp_corvoltd_com__Paint_Item_Request", "altvarname" => "Paint_Item_Request", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PC_Dlvy_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PC_Dlvy_Head", "altvarname" => "PC_Dlvy_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PC_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PC_Head", "altvarname" => "PC_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PC_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PC_Item", "altvarname" => "PC_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_B_Head", "altvarname" => "PI_Imp_B_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_B_Item", "altvarname" => "PI_Imp_B_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Item_Ex", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_B_Item_Ex", "altvarname" => "PI_Imp_B_Item_Ex", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Pack", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_B_Pack", "altvarname" => "PI_Imp_B_Pack", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_C_Head", "altvarname" => "PI_Imp_C_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_C_Item", "altvarname" => "PI_Imp_C_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Pack", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_C_Pack", "altvarname" => "PI_Imp_C_Pack", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Plus_Minus", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_C_Plus_Minus", "altvarname" => "PI_Imp_C_Plus_Minus", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Rmks", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_C_Rmks", "altvarname" => "PI_Imp_C_Rmks", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_R_Head", "altvarname" => "PI_Imp_R_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_R_Item", "altvarname" => "PI_Imp_R_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Pack", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_R_Pack", "altvarname" => "PI_Imp_R_Pack", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Plus_Minus", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_R_Plus_Minus", "altvarname" => "PI_Imp_R_Plus_Minus", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Rmks", "varname" => "firefoxbd_at_erp_corvoltd_com__PI_Imp_R_Rmks", "altvarname" => "PI_Imp_R_Rmks", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_F_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PO_Loc_F_Head", "altvarname" => "PO_Loc_F_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_F_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PO_Loc_F_Item", "altvarname" => "PO_Loc_F_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Pro_Checklist_IBWS", "varname" => "firefoxbd_at_erp_corvoltd_com__Pro_Checklist_IBWS", "altvarname" => "Pro_Checklist_IBWS", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_B_Head", "altvarname" => "PSI_Imp_B_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_B_Item", "altvarname" => "PSI_Imp_B_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Main", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_B_Main", "altvarname" => "PSI_Imp_B_Main", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_plusminus", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_B_plusminus", "altvarname" => "PSI_Imp_B_plusminus", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Remarks", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_B_Remarks", "altvarname" => "PSI_Imp_B_Remarks", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_C_Head", "altvarname" => "PSI_Imp_C_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_C_Item", "altvarname" => "PSI_Imp_C_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Main", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_C_Main", "altvarname" => "PSI_Imp_C_Main", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_plusminus", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_C_plusminus", "altvarname" => "PSI_Imp_C_plusminus", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Remarks", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_C_Remarks", "altvarname" => "PSI_Imp_C_Remarks", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_R_Head", "altvarname" => "PSI_Imp_R_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_R_Item", "altvarname" => "PSI_Imp_R_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Main", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_R_Main", "altvarname" => "PSI_Imp_R_Main", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_plusminus", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_R_plusminus", "altvarname" => "PSI_Imp_R_plusminus", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Remarks", "varname" => "firefoxbd_at_erp_corvoltd_com__PSI_Imp_R_Remarks", "altvarname" => "PSI_Imp_R_Remarks", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Replacement_Item_BD", "varname" => "firefoxbd_at_erp_corvoltd_com__Replacement_Item_BD", "altvarname" => "Replacement_Item_BD", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Sc_Reusable", "varname" => "firefoxbd_at_erp_corvoltd_com__Sc_Reusable", "altvarname" => "Sc_Reusable", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Sp_Kitem", "varname" => "firefoxbd_at_erp_corvoltd_com__Sp_Kitem", "altvarname" => "Sp_Kitem", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Spec_Comp", "varname" => "firefoxbd_at_erp_corvoltd_com__Spec_Comp", "altvarname" => "Spec_Comp", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Spec_Kids_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__Spec_Kids_Head", "altvarname" => "Spec_Kids_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Isu_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_B_Isu_Head", "altvarname" => "ST_B_Isu_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Isu_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_B_Isu_Item", "altvarname" => "ST_B_Isu_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Rec_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_B_Rec_Head", "altvarname" => "ST_B_Rec_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Rec_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_B_Rec_Item", "altvarname" => "ST_B_Rec_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Ret_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_B_Ret_Head", "altvarname" => "ST_B_Ret_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_B_Ret_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_B_Ret_Item", "altvarname" => "ST_B_Ret_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_C_Isu_Head", "altvarname" => "ST_C_Isu_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Head_New", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_C_Isu_Head_New", "altvarname" => "ST_C_Isu_Head_New", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_C_Isu_Item", "altvarname" => "ST_C_Isu_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Isu_Item_New", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_C_Isu_Item_New", "altvarname" => "ST_C_Isu_Item_New", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Rec_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_C_Rec_Head", "altvarname" => "ST_C_Rec_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_C_Rec_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_C_Rec_Item", "altvarname" => "ST_C_Rec_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Rec_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_F_Rec_Head", "altvarname" => "ST_F_Rec_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_F_Rec_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_F_Rec_Item", "altvarname" => "ST_F_Rec_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_Paint_Isu_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__ST_Paint_Isu_Item", "altvarname" => "ST_Paint_Isu_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Temp_SC_Seq_ScPcTTT_Price", "varname" => "firefoxbd_at_erp_corvoltd_com__Temp_SC_Seq_ScPcTTT_Price", "altvarname" => "Temp_SC_Seq_ScPcTTT_Price", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_CusDec", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_CusDec", "altvarname" => "TRI_CusDec", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Exp_Inv_Cont", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Exp_Inv_Cont", "altvarname" => "TRI_Exp_Inv_Cont", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Exp_Inv_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Exp_Inv_Head", "altvarname" => "TRI_Exp_Inv_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Exp_Inv_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Exp_Inv_Item", "altvarname" => "TRI_Exp_Inv_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Exp_Inv_Main", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Exp_Inv_Main", "altvarname" => "TRI_Exp_Inv_Main", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_FGS_Actual_Receipt", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_FGS_Actual_Receipt", "altvarname" => "TRI_FGS_Actual_Receipt", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Loading_Head_New", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Loading_Head_New", "altvarname" => "TRI_Loading_Head_New", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Loading_Item_New", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Loading_Item_New", "altvarname" => "TRI_Loading_Item_New", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Model_Maser", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Model_Maser", "altvarname" => "TRI_Model_Maser", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_B_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_B_Head", "altvarname" => "TRI_PI_Imp_B_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_B_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_B_Item", "altvarname" => "TRI_PI_Imp_B_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_B_Pack", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_B_Pack", "altvarname" => "TRI_PI_Imp_B_Pack", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_R_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_R_Head", "altvarname" => "TRI_PI_Imp_R_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_R_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_R_Item", "altvarname" => "TRI_PI_Imp_R_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_R_Pack", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_R_Pack", "altvarname" => "TRI_PI_Imp_R_Pack", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_R_Plus_Minus", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_R_Plus_Minus", "altvarname" => "TRI_PI_Imp_R_Plus_Minus", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PI_Imp_R_Rmks", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PI_Imp_R_Rmks", "altvarname" => "TRI_PI_Imp_R_Rmks", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PSI_Imp_A_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PSI_Imp_A_Item", "altvarname" => "TRI_PSI_Imp_A_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PSI_Imp_B_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PSI_Imp_B_Head", "altvarname" => "TRI_PSI_Imp_B_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PSI_Imp_B_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PSI_Imp_B_Item", "altvarname" => "TRI_PSI_Imp_B_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PSI_Imp_B_Main", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PSI_Imp_B_Main", "altvarname" => "TRI_PSI_Imp_B_Main", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PSI_Imp_B_plusminus", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PSI_Imp_B_plusminus", "altvarname" => "TRI_PSI_Imp_B_plusminus", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PSI_Imp_B_Remarks", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PSI_Imp_B_Remarks", "altvarname" => "TRI_PSI_Imp_B_Remarks", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_PSI_Imp_R_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_PSI_Imp_R_Item", "altvarname" => "TRI_PSI_Imp_R_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Replacement_Item_BD", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Replacement_Item_BD", "altvarname" => "TRI_Replacement_Item_BD", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_B_Isu_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_B_Isu_Head", "altvarname" => "TRI_ST_B_Isu_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_B_Isu_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_B_Isu_Item", "altvarname" => "TRI_ST_B_Isu_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_B_Rec_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_B_Rec_Head", "altvarname" => "TRI_ST_B_Rec_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_B_Rec_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_B_Rec_Item", "altvarname" => "TRI_ST_B_Rec_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_B_Ret_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_B_Ret_Head", "altvarname" => "TRI_ST_B_Ret_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_B_Ret_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_B_Ret_Item", "altvarname" => "TRI_ST_B_Ret_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_C_Isu_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_C_Isu_Head", "altvarname" => "TRI_ST_C_Isu_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_C_Isu_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_C_Isu_Item", "altvarname" => "TRI_ST_C_Isu_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_C_Rec_Head", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_C_Rec_Head", "altvarname" => "TRI_ST_C_Rec_Head", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_ST_C_Rec_Item", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_ST_C_Rec_Item", "altvarname" => "TRI_ST_C_Rec_Item", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "TRI_Welding_Wip_Head_AL", "varname" => "firefoxbd_at_erp_corvoltd_com__TRI_Welding_Wip_Head_AL", "altvarname" => "TRI_Welding_Wip_Head_AL", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "viud1_settings", "varname" => "firefoxbd_at_erp_corvoltd_com__viud1_settings", "altvarname" => "viud1_settings", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "viud1_uggroups", "varname" => "firefoxbd_at_erp_corvoltd_com__viud1_uggroups", "altvarname" => "viud1_uggroups", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "viud1_ugmembers", "varname" => "firefoxbd_at_erp_corvoltd_com__viud1_ugmembers", "altvarname" => "viud1_ugmembers", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "viud1_ugrights", "varname" => "firefoxbd_at_erp_corvoltd_com__viud1_ugrights", "altvarname" => "viud1_ugrights", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "viudcorvo_audit", "varname" => "firefoxbd_at_erp_corvoltd_com__viudcorvo_audit", "altvarname" => "viudcorvo_audit", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "webusers", "varname" => "firefoxbd_at_erp_corvoltd_com__webusers", "altvarname" => "webusers", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Week_No_Master", "varname" => "firefoxbd_at_erp_corvoltd_com__Week_No_Master", "altvarname" => "Week_No_Master", "connId" => "firefoxbd_at_erp_corvoltd_com", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>