<?php
class DB
{
	public static function CurrentConnection()
	{
		global $currentConnection;
		return $currentConnection ? $currentConnection : DB::DefaultConnection();
	}

	public static function CurrentConnectionId()
	{
		$conn = DB::CurrentConnection();
		return $conn->connId;
	}

	public static function DefaultConnection()
	{
		global $cman;
		return $cman->getDefault();
	}

	public static function ConnectionByTable( $table )
	{
		global $cman;
		return $cman->byTable($table);
	}

	public static function ConnectionByName( $name )
	{
		global $cman;
		return $cman->byName( $name );
	}

	public static function SetConnection( $connection )
	{
		global $currentConnection;
		if ( is_string( $connection ) )
		{
			$currentConnection = DB::ConnectionByName( $connection );
		}
		else if ( is_a($connection, 'Connection') ) {
		 	$currentConnection = $connection;
		}
	}

	public static function LastId()
	{
		return DB::CurrentConnection()->getInsertedId();
	}

	public static function Query( $sql )
	{
		return DB::CurrentConnection()->querySilent( $sql );
	}

	public static function Exec( $sql )
	{
		return DB::CurrentConnection()->execSilent( $sql ) != NULL;
	}

	public static function LastError()
	{
		return DB::CurrentConnection()->lastError();
	}

	public static function Select($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		$sql = "SELECT * FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$queryResult = DB::CurrentConnection()->querySilent( $sql );

		return $queryResult;
	}

	public static function Delete($table, $userConditions = array() )
	{
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$sql = "DELETE FROM ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] ) . $whereSql;
		$ret = DB::CurrentConnection()->execSilent( $sql );

		return $ret;
	}

	public static function Insert($table, $data)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$iFields = "";
		$iValues = "";
		$blobs = array();
		foreach($data as $fieldname => $value)
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$iFields.= DB::CurrentConnection()->addFieldWrappers( $field["name"] ).",";
			$iValues.= DB::_prepareValue($value, $field["type"]) . ",";

			if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if( $iFields != "" && $iValues != "" )
		{
			$iFields = substr($iFields, 0, -1);
			$iValues = substr($iValues, 0, -1);
			$sql = "INSERT INTO ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." (".$iFields.") values (".$iValues.")";

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	public static function Update($table, $data, $userConditions)
	{
		$result = false;
		$tableInfo = DB::_getTableInfo($table);

		if ( !$tableInfo )
			return false;

		$whereSql = DB::_getWhereSql($userConditions, $tableInfo["fields"]);

		if( $whereSql == "" )
			return false;

		$updateValues = array();
		$blobs = array();
		foreach( $data as $fieldname => $value )
		{
			$field = getArrayElementNC($tableInfo["fields"], $fieldname);

			// user field not found in table
			if ( is_null($field) )
				continue;

			$prepareFieldName = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			$prepareValue = DB::_prepareValue($value, $field["type"]);
			$updateValues[] = $prepareFieldName."=".$prepareValue;

			if ( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
			{
				if ( IsBinaryType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;

				if ( DB::CurrentConnection()->dbType == nDATABASE_Informix && IsTextType( $field["type"] ) )
					$blobs[ $field["name"] ] = $value;
			}
		}

		if ( count($updateValues) > 0 )
		{
			$updateSQL = implode(",", $updateValues);
			$sql = "UPDATE ".DB::CurrentConnection()->addTableWrappers( $tableInfo['fullName'] )." SET ". $updateSQL . $whereSql;

			if ( count($blobs) > 0 )
				$result = DB::_execSilentWithBlobProcessing($blobs, $sql, $tableInfo['fields']);
			else
				$result = DB::CurrentConnection()->execSilent( $sql );
		}

		return $result;
	}

	protected static function _getWhereSql($userConditions, $founedfields)
	{
		if( !is_array( $userConditions ) )
		{
			$whereSql = trim( $userConditions );
			if( $whereSql != "")
				$whereSql = " WHERE " . $whereSql;
			return $whereSql;
		}

		$conditions = array();
		foreach($userConditions as $fieldname => $value)
		{
			$field = getArrayElementNC($founedfields, $fieldname);
			// user field not found in table
			if ( is_null($field) )
				continue;

			$wrappedField = DB::CurrentConnection()->addFieldWrappers( $field["name"] );
			if ( is_null($value) )
			{
				$conditions[] = $wrappedField . " IS NULL";
			}
			else
			{
				$conditions[] = $wrappedField . "=" . DB::_prepareValue($value, $field["type"]);
			}
		}

		$whereSql = "";
		if( count($conditions) > 0 )
		{
			$whereSql .= " WHERE " . implode(" AND ", $conditions);
		}

		return $whereSql;
	}

	/**
	 * @param Array blobs
	 * @param String dalSQL
	 * @param Array tableinfo
	 */
	protected static function _execSilentWithBlobProcessing($blobs, $dalSQL, $tableinfo)
	{
		$blobTypes = array();
		if( DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			foreach( $blobs as $fname => $fvalue )
			{
				$blobTypes[ $fname ] = $tableinfo[ $fname ]["type"];
			}
		}

		DB::CurrentConnection()->execSilentWithBlobProcessing( $dalSQL, $blobs, $blobTypes );
	}

	protected static function _prepareValue($value, $type)
	{
		if ( is_null($value) )
			return "NULL";

		if( DB::CurrentConnection()->dbType == nDATABASE_Oracle || DB::CurrentConnection()->dbType == nDATABASE_DB2 || DB::CurrentConnection()->dbType == nDATABASE_Informix )
		{
			if( IsBinaryType($type) )
			{
				if( DB::CurrentConnection()->dbType == nDATABASE_Oracle )
					return "EMPTY_BLOB()";

				return "?";
			}

			if( DB::CurrentConnection()->dbType == nDATABASE_Informix  && IsTextType($type) )
				return "?";
		}

		if( IsNumberType($type) && !is_numeric($value) )
		{
			$value = trim($value);
			$value = str_replace(",", ".", $value);
			if ( !is_numeric($value) )
				return "NULL";
		}

		if( IsDateFieldType($type) || IsTimeType($type) )
		{
			if( !$value )
				return "NULL";

			// timestamp
			if ( is_int($value) )
			{
				if ( IsDateFieldType($type) )
				{
					$value = getYMDdate($value) . " " . getHISdate($value);
				}
				else if ( IsTimeType($type) )
				{
					$value = getHISdate($value);
				}
			}

			return DB::CurrentConnection()->addDateQuotes( $value );
		}

		if( NeedQuotes($type) )
			return DB::CurrentConnection()->prepareString( $value );

		return $value;
	}

	/**
	 * 	Find table info stored in the project file
	 *
	 */
	protected static function _findDalTable( $table )
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		$tableName = $conn->getTableNameComponents( $table );

		DB::_fillTablesList();

		//	exact match
		foreach( $dalTables[$conn->connId] as $t )
		{
			if( $t["schema"] == $tableName["schema"] && $t["name"] == $tableName["table"] )
				return $t;
		}

		//	case-insensitive
		$tableName["schema"] = strtoupper( $tableName["schema"] );
		$tableName["table"] = strtoupper( $tableName["table"] );

		foreach( $dalTables[$conn->connId] as $t )
		{
			if( strtoupper( $t["schema"] ) == $tableName["schema"] && strtoupper( $t["name"] ) == $tableName["table"] )
				return $t;
		}
		return null;
	}

	/**
	 * 	Get list of table field names and types
	 *	Check tables stored in the project first, then fetch it from the database.
	 *
	 */
	protected static function _getTableInfo($table)
	{
		global $dal_info, $tableinfo_cache;
		$connId = DB::CurrentConnectionId();

		//	prepare cache
		if( !isset($tableinfo_cache[ $connId ] ) )
			$tableinfo_cache[ $connId ] = array();

		$tableInfo = array();


		$tableDescriptor = DB::_findDalTable( $table );

		if ( $tableDescriptor )
		{
			importTableInfo( $tableDescriptor["varname"] );

			$tableInfo["fields"] = $dal_info[ $tableDescriptor["varname"] ];

			if( $tableDescriptor["schema"] )
				$tableInfo["fullName"] = $tableDescriptor["schema"] . "." . $tableDescriptor["name"];
			else
				$tableInfo["fullName"] = $tableDescriptor["name"];
		}
		else
		{
			//	check cache first
			if( isset($tableinfo_cache[ $connId ][ $table ] ) )
				return $tableinfo_cache[ $connId ][ $table ];

			//	fetch table info from the database
			$helpSql = "select * from " . DB::CurrentConnection()->addTableWrappers( $table ) . " where 1=0";

			$tableInfo["fullName"] = $table;
			$tableInfo["fields"] = array();

			// in case getFieldsList throws error
			$tableinfo_cache[ $connId ][ $table ] = false;

			$fieldList = DB::CurrentConnection()->getFieldsList($helpSql);
			foreach ($fieldList as $f )
			{
				$tableInfo["fields"][ $f["fieldname"] ] = array( "type" => $f["type"], "name" => $f["fieldname"] );
			}
			$tableinfo_cache[ $connId ][ $table ] = $tableInfo;
		}

		return $tableInfo;
	}


	protected static function _fillTablesList()
	{
		global $dalTables;
		$conn = DB::CurrentConnection();
		if( $dalTables[ $conn->connId ] )
			return;
		$dalTables[ $conn->connId ] = array();
		if( "firefoxbd_at_localhost" == $conn->connId )
		{
			$dalTables[$conn->connId][] = array("name" => "Agent_Master", "varname" => "firefoxbd_at_localhost__Agent_Master", "altvarname" => "Agent_Master", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Assets_Group", "varname" => "firefoxbd_at_localhost__Assets_Group", "altvarname" => "Assets_Group", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Assets_Item", "varname" => "firefoxbd_at_localhost__Assets_Item", "altvarname" => "Assets_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "BD_Model", "varname" => "firefoxbd_at_localhost__BD_Model", "altvarname" => "BD_Model", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "BL_Shipper", "varname" => "firefoxbd_at_localhost__BL_Shipper", "altvarname" => "BL_Shipper", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "corvoproject_audit", "varname" => "firefoxbd_at_localhost__corvoproject_audit", "altvarname" => "corvoproject_audit", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Country", "varname" => "firefoxbd_at_localhost__Country", "altvarname" => "Country", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Currency", "varname" => "firefoxbd_at_localhost__Currency", "altvarname" => "Currency", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Customer", "varname" => "firefoxbd_at_localhost__Customer", "altvarname" => "Customer", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Destinations", "varname" => "firefoxbd_at_localhost__Destinations", "altvarname" => "Destinations", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Factory", "varname" => "firefoxbd_at_localhost__Factory", "altvarname" => "Factory", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_uggroups", "varname" => "firefoxbd_at_localhost__ffxbderp_uggroups", "altvarname" => "ffxbderp_uggroups", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugmembers", "varname" => "firefoxbd_at_localhost__ffxbderp_ugmembers", "altvarname" => "ffxbderp_ugmembers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ffxbderp_ugrights", "varname" => "firefoxbd_at_localhost__ffxbderp_ugrights", "altvarname" => "ffxbderp_ugrights", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Forwaders", "varname" => "firefoxbd_at_localhost__Forwaders", "altvarname" => "Forwaders", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Fright", "varname" => "firefoxbd_at_localhost__Fright", "altvarname" => "Fright", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Fright_new", "varname" => "firefoxbd_at_localhost__Fright_new", "altvarname" => "Fright_new", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Head_Inv", "varname" => "firefoxbd_at_localhost__Ibws_Head_Inv", "altvarname" => "Ibws_Head_Inv", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Inv_Plus_Minus", "varname" => "firefoxbd_at_localhost__Ibws_Inv_Plus_Minus", "altvarname" => "Ibws_Inv_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ibws_Inv_Remarks", "varname" => "firefoxbd_at_localhost__Ibws_Inv_Remarks", "altvarname" => "Ibws_Inv_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Imp_A_Remarks", "varname" => "firefoxbd_at_localhost__Imp_A_Remarks", "altvarname" => "Imp_A_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Imp_B_Remarks", "varname" => "firefoxbd_at_localhost__Imp_B_Remarks", "altvarname" => "Imp_B_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Imp_C_Remarks", "varname" => "firefoxbd_at_localhost__Imp_C_Remarks", "altvarname" => "Imp_C_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Imp_Paint_Remarks", "varname" => "firefoxbd_at_localhost__Imp_Paint_Remarks", "altvarname" => "Imp_Paint_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Imp_R_Remarks", "varname" => "firefoxbd_at_localhost__Imp_R_Remarks", "altvarname" => "Imp_R_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Imp_Rc_Remarks", "varname" => "firefoxbd_at_localhost__Imp_Rc_Remarks", "altvarname" => "Imp_Rc_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Item_Unit", "varname" => "firefoxbd_at_localhost__Item_Unit", "altvarname" => "Item_Unit", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Loading_Ports", "varname" => "firefoxbd_at_localhost__Loading_Ports", "altvarname" => "Loading_Ports", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Maintenance_Input", "varname" => "firefoxbd_at_localhost__Maintenance_Input", "altvarname" => "Maintenance_Input", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Maintenance_Item", "varname" => "firefoxbd_at_localhost__Maintenance_Item", "altvarname" => "Maintenance_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Model_Group", "varname" => "firefoxbd_at_localhost__Model_Group", "altvarname" => "Model_Group", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Paint_Head", "varname" => "firefoxbd_at_localhost__Paint_Head", "altvarname" => "Paint_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Paint_Item", "varname" => "firefoxbd_at_localhost__Paint_Item", "altvarname" => "Paint_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Parts", "varname" => "firefoxbd_at_localhost__Parts", "altvarname" => "Parts", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Payterms_new", "varname" => "firefoxbd_at_localhost__Payterms_new", "altvarname" => "Payterms_new", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PC_Head", "varname" => "firefoxbd_at_localhost__PC_Head", "altvarname" => "PC_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PC_Item", "varname" => "firefoxbd_at_localhost__PC_Item", "altvarname" => "PC_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Pc_Terms", "varname" => "firefoxbd_at_localhost__Pc_Terms", "altvarname" => "Pc_Terms", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_A_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_A_Head", "altvarname" => "PI_Imp_A_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_A_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_A_Item", "altvarname" => "PI_Imp_A_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_A_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_A_Pack", "altvarname" => "PI_Imp_A_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_A_Plus_Minus", "varname" => "firefoxbd_at_localhost__PI_Imp_A_Plus_Minus", "altvarname" => "PI_Imp_A_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_A_Rmks", "varname" => "firefoxbd_at_localhost__PI_Imp_A_Rmks", "altvarname" => "PI_Imp_A_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_B_Head", "altvarname" => "PI_Imp_B_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_B_Item", "altvarname" => "PI_Imp_B_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Item_Ex", "varname" => "firefoxbd_at_localhost__PI_Imp_B_Item_Ex", "altvarname" => "PI_Imp_B_Item_Ex", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_B_Pack", "altvarname" => "PI_Imp_B_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Plus_Minus", "varname" => "firefoxbd_at_localhost__PI_Imp_B_Plus_Minus", "altvarname" => "PI_Imp_B_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_B_Rmks", "varname" => "firefoxbd_at_localhost__PI_Imp_B_Rmks", "altvarname" => "PI_Imp_B_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_C_Head", "altvarname" => "PI_Imp_C_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_C_Item", "altvarname" => "PI_Imp_C_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_C_Pack", "altvarname" => "PI_Imp_C_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Plus_Minus", "varname" => "firefoxbd_at_localhost__PI_Imp_C_Plus_Minus", "altvarname" => "PI_Imp_C_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_C_Rmks", "varname" => "firefoxbd_at_localhost__PI_Imp_C_Rmks", "altvarname" => "PI_Imp_C_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Paint_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_Paint_Head", "altvarname" => "PI_Imp_Paint_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Paint_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_Paint_Item", "altvarname" => "PI_Imp_Paint_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Paint_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_Paint_Pack", "altvarname" => "PI_Imp_Paint_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Paint_Plus_Minus", "varname" => "firefoxbd_at_localhost__PI_Imp_Paint_Plus_Minus", "altvarname" => "PI_Imp_Paint_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Paint_Rmks", "varname" => "firefoxbd_at_localhost__PI_Imp_Paint_Rmks", "altvarname" => "PI_Imp_Paint_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_R_Head", "altvarname" => "PI_Imp_R_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_R_Item", "altvarname" => "PI_Imp_R_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_R_Pack", "altvarname" => "PI_Imp_R_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Plus_Minus", "varname" => "firefoxbd_at_localhost__PI_Imp_R_Plus_Minus", "altvarname" => "PI_Imp_R_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_R_Rmks", "varname" => "firefoxbd_at_localhost__PI_Imp_R_Rmks", "altvarname" => "PI_Imp_R_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rc_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_Rc_Head", "altvarname" => "PI_Imp_Rc_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rc_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_Rc_Item", "altvarname" => "PI_Imp_Rc_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rc_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_Rc_Pack", "altvarname" => "PI_Imp_Rc_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rc_Plus_Minus", "varname" => "firefoxbd_at_localhost__PI_Imp_Rc_Plus_Minus", "altvarname" => "PI_Imp_Rc_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rc_Rmks", "varname" => "firefoxbd_at_localhost__PI_Imp_Rc_Rmks", "altvarname" => "PI_Imp_Rc_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rim_Head", "varname" => "firefoxbd_at_localhost__PI_Imp_Rim_Head", "altvarname" => "PI_Imp_Rim_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rim_Item", "varname" => "firefoxbd_at_localhost__PI_Imp_Rim_Item", "altvarname" => "PI_Imp_Rim_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PI_Imp_Rim_Pack", "varname" => "firefoxbd_at_localhost__PI_Imp_Rim_Pack", "altvarname" => "PI_Imp_Rim_Pack", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_A_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_A_Head", "altvarname" => "PO_Imp_A_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_A_Item", "varname" => "firefoxbd_at_localhost__PO_Imp_A_Item", "altvarname" => "PO_Imp_A_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_A_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Imp_A_Plus_Minus", "altvarname" => "PO_Imp_A_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_B_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_B_Head", "altvarname" => "PO_Imp_B_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_B_Item", "varname" => "firefoxbd_at_localhost__PO_Imp_B_Item", "altvarname" => "PO_Imp_B_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_B_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Imp_B_Plus_Minus", "altvarname" => "PO_Imp_B_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_C_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_C_Head", "altvarname" => "PO_Imp_C_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_C_Item", "varname" => "firefoxbd_at_localhost__PO_Imp_C_Item", "altvarname" => "PO_Imp_C_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_C_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Imp_C_Plus_Minus", "altvarname" => "PO_Imp_C_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_F_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_F_Head", "altvarname" => "PO_Imp_F_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_F_Item", "varname" => "firefoxbd_at_localhost__PO_Imp_F_Item", "altvarname" => "PO_Imp_F_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_F_ItemData", "varname" => "firefoxbd_at_localhost__PO_Imp_F_ItemData", "altvarname" => "PO_Imp_F_ItemData", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_R_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_R_Head", "altvarname" => "PO_Imp_R_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_R_Item", "varname" => "firefoxbd_at_localhost__PO_Imp_R_Item", "altvarname" => "PO_Imp_R_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_R_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Imp_R_Plus_Minus", "altvarname" => "PO_Imp_R_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_Rc_Head", "varname" => "firefoxbd_at_localhost__PO_Imp_Rc_Head", "altvarname" => "PO_Imp_Rc_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_Rc_Item", "varname" => "firefoxbd_at_localhost__PO_Imp_Rc_Item", "altvarname" => "PO_Imp_Rc_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Imp_Rc_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Imp_Rc_Plus_Minus", "altvarname" => "PO_Imp_Rc_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_A_Head", "varname" => "firefoxbd_at_localhost__PO_Loc_A_Head", "altvarname" => "PO_Loc_A_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_A_Item", "varname" => "firefoxbd_at_localhost__PO_Loc_A_Item", "altvarname" => "PO_Loc_A_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_A_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Loc_A_Plus_Minus", "altvarname" => "PO_Loc_A_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_A_Rmks", "varname" => "firefoxbd_at_localhost__PO_Loc_A_Rmks", "altvarname" => "PO_Loc_A_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_B_Head", "varname" => "firefoxbd_at_localhost__PO_Loc_B_Head", "altvarname" => "PO_Loc_B_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_B_Item", "varname" => "firefoxbd_at_localhost__PO_Loc_B_Item", "altvarname" => "PO_Loc_B_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_B_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Loc_B_Plus_Minus", "altvarname" => "PO_Loc_B_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_B_Rmks", "varname" => "firefoxbd_at_localhost__PO_Loc_B_Rmks", "altvarname" => "PO_Loc_B_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_C_Head", "varname" => "firefoxbd_at_localhost__PO_Loc_C_Head", "altvarname" => "PO_Loc_C_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_C_Item", "varname" => "firefoxbd_at_localhost__PO_Loc_C_Item", "altvarname" => "PO_Loc_C_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_C_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Loc_C_Plus_Minus", "altvarname" => "PO_Loc_C_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_C_Rmks", "varname" => "firefoxbd_at_localhost__PO_Loc_C_Rmks", "altvarname" => "PO_Loc_C_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_F_Head", "varname" => "firefoxbd_at_localhost__PO_Loc_F_Head", "altvarname" => "PO_Loc_F_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_F_Item", "varname" => "firefoxbd_at_localhost__PO_Loc_F_Item", "altvarname" => "PO_Loc_F_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_F_Plus_Minus", "varname" => "firefoxbd_at_localhost__PO_Loc_F_Plus_Minus", "altvarname" => "PO_Loc_F_Plus_Minus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_Loc_F_Rmks", "varname" => "firefoxbd_at_localhost__PO_Loc_F_Rmks", "altvarname" => "PO_Loc_F_Rmks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_A", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_A", "altvarname" => "PO_PSI_Imp_A", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_A_List", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_A_List", "altvarname" => "PO_PSI_Imp_A_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_B", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_B", "altvarname" => "PO_PSI_Imp_B", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_B_List", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_B_List", "altvarname" => "PO_PSI_Imp_B_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_C", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_C", "altvarname" => "PO_PSI_Imp_C", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_C_List", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_C_List", "altvarname" => "PO_PSI_Imp_C_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_Paint", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_Paint", "altvarname" => "PO_PSI_Imp_Paint", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_Paint_List", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_Paint_List", "altvarname" => "PO_PSI_Imp_Paint_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_R", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_R", "altvarname" => "PO_PSI_Imp_R", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_R_List", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_R_List", "altvarname" => "PO_PSI_Imp_R_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_Rc", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_Rc", "altvarname" => "PO_PSI_Imp_Rc", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PO_PSI_Imp_Rc_List", "varname" => "firefoxbd_at_localhost__PO_PSI_Imp_Rc_List", "altvarname" => "PO_PSI_Imp_Rc_List", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_Head", "varname" => "firefoxbd_at_localhost__PSI_Imp_A_Head", "altvarname" => "PSI_Imp_A_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_Item", "varname" => "firefoxbd_at_localhost__PSI_Imp_A_Item", "altvarname" => "PSI_Imp_A_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_Main", "varname" => "firefoxbd_at_localhost__PSI_Imp_A_Main", "altvarname" => "PSI_Imp_A_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_plusminus", "varname" => "firefoxbd_at_localhost__PSI_Imp_A_plusminus", "altvarname" => "PSI_Imp_A_plusminus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_A_Remarks", "varname" => "firefoxbd_at_localhost__PSI_Imp_A_Remarks", "altvarname" => "PSI_Imp_A_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Head", "varname" => "firefoxbd_at_localhost__PSI_Imp_B_Head", "altvarname" => "PSI_Imp_B_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Item", "varname" => "firefoxbd_at_localhost__PSI_Imp_B_Item", "altvarname" => "PSI_Imp_B_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Main", "varname" => "firefoxbd_at_localhost__PSI_Imp_B_Main", "altvarname" => "PSI_Imp_B_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_plusminus", "varname" => "firefoxbd_at_localhost__PSI_Imp_B_plusminus", "altvarname" => "PSI_Imp_B_plusminus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_B_Remarks", "varname" => "firefoxbd_at_localhost__PSI_Imp_B_Remarks", "altvarname" => "PSI_Imp_B_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Head", "varname" => "firefoxbd_at_localhost__PSI_Imp_C_Head", "altvarname" => "PSI_Imp_C_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Item", "varname" => "firefoxbd_at_localhost__PSI_Imp_C_Item", "altvarname" => "PSI_Imp_C_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Main", "varname" => "firefoxbd_at_localhost__PSI_Imp_C_Main", "altvarname" => "PSI_Imp_C_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_plusminus", "varname" => "firefoxbd_at_localhost__PSI_Imp_C_plusminus", "altvarname" => "PSI_Imp_C_plusminus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_C_Remarks", "varname" => "firefoxbd_at_localhost__PSI_Imp_C_Remarks", "altvarname" => "PSI_Imp_C_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_Head", "varname" => "firefoxbd_at_localhost__PSI_Imp_Paint_Head", "altvarname" => "PSI_Imp_Paint_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_Item", "varname" => "firefoxbd_at_localhost__PSI_Imp_Paint_Item", "altvarname" => "PSI_Imp_Paint_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_Main", "varname" => "firefoxbd_at_localhost__PSI_Imp_Paint_Main", "altvarname" => "PSI_Imp_Paint_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_plusminus", "varname" => "firefoxbd_at_localhost__PSI_Imp_Paint_plusminus", "altvarname" => "PSI_Imp_Paint_plusminus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Paint_Remarks", "varname" => "firefoxbd_at_localhost__PSI_Imp_Paint_Remarks", "altvarname" => "PSI_Imp_Paint_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Head", "varname" => "firefoxbd_at_localhost__PSI_Imp_R_Head", "altvarname" => "PSI_Imp_R_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Item", "varname" => "firefoxbd_at_localhost__PSI_Imp_R_Item", "altvarname" => "PSI_Imp_R_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Main", "varname" => "firefoxbd_at_localhost__PSI_Imp_R_Main", "altvarname" => "PSI_Imp_R_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_plusminus", "varname" => "firefoxbd_at_localhost__PSI_Imp_R_plusminus", "altvarname" => "PSI_Imp_R_plusminus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_R_Remarks", "varname" => "firefoxbd_at_localhost__PSI_Imp_R_Remarks", "altvarname" => "PSI_Imp_R_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_Head", "varname" => "firefoxbd_at_localhost__PSI_Imp_Rc_Head", "altvarname" => "PSI_Imp_Rc_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_Item", "varname" => "firefoxbd_at_localhost__PSI_Imp_Rc_Item", "altvarname" => "PSI_Imp_Rc_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_Main", "varname" => "firefoxbd_at_localhost__PSI_Imp_Rc_Main", "altvarname" => "PSI_Imp_Rc_Main", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_plusminus", "varname" => "firefoxbd_at_localhost__PSI_Imp_Rc_plusminus", "altvarname" => "PSI_Imp_Rc_plusminus", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "PSI_Imp_Rc_Remarks", "varname" => "firefoxbd_at_localhost__PSI_Imp_Rc_Remarks", "altvarname" => "PSI_Imp_Rc_Remarks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "RCConsignee", "varname" => "firefoxbd_at_localhost__RCConsignee", "altvarname" => "RCConsignee", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ref_Replace_Polnvlist", "varname" => "firefoxbd_at_localhost__Ref_Replace_Polnvlist", "altvarname" => "Ref_Replace_Polnvlist", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ref_Replacement_Invlist1", "varname" => "firefoxbd_at_localhost__Ref_Replacement_Invlist1", "altvarname" => "Ref_Replacement_Invlist1", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Ref_Sc_Iniinput", "varname" => "firefoxbd_at_localhost__Ref_Sc_Iniinput", "altvarname" => "Ref_Sc_Iniinput", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Replacement_Item", "varname" => "firefoxbd_at_localhost__Replacement_Item", "altvarname" => "Replacement_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Replcement_Req_Head", "varname" => "firefoxbd_at_localhost__Replcement_Req_Head", "altvarname" => "Replcement_Req_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Sc_Head", "varname" => "firefoxbd_at_localhost__Sc_Head", "altvarname" => "Sc_Head", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "SC_Item", "varname" => "firefoxbd_at_localhost__SC_Item", "altvarname" => "SC_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Shipping_Marks", "varname" => "firefoxbd_at_localhost__Shipping_Marks", "altvarname" => "Shipping_Marks", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Spec_Group", "varname" => "firefoxbd_at_localhost__Spec_Group", "altvarname" => "Spec_Group", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_Item", "varname" => "firefoxbd_at_localhost__ST_Item", "altvarname" => "ST_Item", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Bulk", "varname" => "firefoxbd_at_localhost__ST_Item_Bulk", "altvarname" => "ST_Item_Bulk", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Carton", "varname" => "firefoxbd_at_localhost__ST_Item_Carton", "altvarname" => "ST_Item_Carton", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "ST_Item_Chain", "varname" => "firefoxbd_at_localhost__ST_Item_Chain", "altvarname" => "ST_Item_Chain", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Supplier", "varname" => "firefoxbd_at_localhost__Supplier", "altvarname" => "Supplier", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Supplier_Maintenance", "varname" => "firefoxbd_at_localhost__Supplier_Maintenance", "altvarname" => "Supplier_Maintenance", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Time_Period", "varname" => "firefoxbd_at_localhost__Time_Period", "altvarname" => "Time_Period", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "Title", "varname" => "firefoxbd_at_localhost__Title", "altvarname" => "Title", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
			$dalTables[$conn->connId][] = array("name" => "webusers", "varname" => "firefoxbd_at_localhost__webusers", "altvarname" => "webusers", "connId" => "firefoxbd_at_localhost", "schema" => "", "connName" => "firefoxbd at erp.corvoltd.com");
		}
	}


	public static function PrepareSQL($sql)
	{
		$args = func_get_args();

		$conn = DB::CurrentConnection();
		$context = RunnerContext::current();

		$tokens = DB::scanTokenString($sql);

		$replacements = array();
		// build array of replacements in this format:
		//	"offset" => position in the string where replacement should be done
		//  "len" => length of original substring to cut out
		//  "insert" => string to insert in place of cut out

		foreach ($tokens["matches"] as $i => $match) {
			$offset = $tokens["offsets"][$i];
			$token = $tokens["tokens"][$i];

			$repl = array(
				"offset" => $offset,
				"len" => strlen($match)
			);

			$val = "";
			if (is_numeric($token) && count( $args ) > $token) {
				$val = $args[(int)$token];
			} else {
				$val = $context->getValue($token);
			}

			if ($conn->positionQuoted($sql, $offset))
				$repl["insert"] = $conn->addSlashes($val);
			else if(is_numeric($val))
				$repl["insert"] = DB::prepareNumberValue($val);
			else
				$repl["insert"] = $val;

			$replacements[] = $repl;
		}

		//	do replacements
		$offsetShift = 0;
		foreach ($replacements as $r) {
			$sql = substr_replace($sql, $r["insert"], $r["offset"] + $offsetShift, $r["len"]);
			$offsetShift += strlen($r["insert"]) - $r["len"];
		}
		return $sql;
	}

	/**
	 *	@return Array
	 */
	public static function readSQLTokens( $sql )
	{
		$arr = DB::scanTokenString( $sql );
		return $arr["tokens"];
	}

	/**
	 *	@return Array
	 */
	public static function readMasterTokens( $sql )
	{
		$masterTokens = array();

		$allTokens = DB::readSQLTokens( $sql );
		foreach ( $allTokens as $key => $token )
		{
			$dotPos = strpos(  $token, "." );
			if( $dotPos !== FALSE && strtolower( substr( $token, 0, $dotPos ) ) == "master")
			{
				$masterTokens[] = $token;
			}
		}

		return $masterTokens;
	}

	/**
	 *	Scans SQL string, finds all tokens. Returns three arrays - 'tokens', 'matches' and 'offsets'
	 *  Offsets are positions of corresponding 'matches' items in the string
	 *  Example:
	 *  insert into table values (':aaa', :old.bbb, ':{master.order date}')
	 *  tokens: ["aaa", "old.bbb", "master.order date"]
	 *  matches: [":aaa", ":old.bbb", ":{master.order date}"]
	 *  offsets: [28, 35, 46]
	 *
	 *	Exceptions for tokens without {}
	 *	1. shouldn't start with number
	*		:62aaa
	 *	2. shouldn't follow letter
	 *		x:aaa
	 *	3. shouldn't follow :
	 *		::aaa
	 *
 	 *	@return Array [ "tokens" => Array, "matches" => Array, "offsets" => Array ]
	 */
	protected static function scanTokenString($sql)
	{
		$tokens = array();
		$offsets = array();
		$matches = array();

		//	match aaa, old.bbb, master.order date from:
		//	insert into table values (':aaa', :old.bbb, ':{master.order date}')

		$pattern = '/(?:[^\w\:]|^)(\:([a-zA-Z_]{1}[\w\.]*))|\:\{(.*?)\}|\:([1-9]+[0-9]*)/';


		$result = findMatches($pattern, $sql);
		foreach ($result as $m) {
			if ($m["submatches"][0] != "") {
				//	first variant, no {}
				$matches[] = $m["submatches"][0];
				$tokens[] = $m["submatches"][1];
				$offsets[] = $m["offset"] + strpos($m["match"], $m["submatches"][0]);
			} else if ($m["submatches"][2] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][2];
				$offsets[] = $m["offset"];
			} else if ($m["submatches"][3] != "") {
				$matches[] = $m["match"];
				$tokens[] = $m["submatches"][3];
				$offsets[] = $m["offset"];
			}
		}

		return array("tokens" => $tokens, "matches" => $matches, "offsets" => $offsets);
	}


	public static function prepareNumberValue( $value )
	{
		$strvalue = (string)$value;
		if(is_numeric($strvalue))
			return str_replace(",",".",$strvalue);
		return 0;
	}

}

?>